<!-- OTP Verification Modal -->
<div class="modal fade" id="otpModal" tabindex="-1" role="dialog" aria-labelledby="otpModalLabel" aria-hidden="true" data-backdrop="static" data-keyboard="false" style="display: none;">
    <div class="modal-dialog modal-dialog-centered" role="document">
        <div class="modal-content border-0 shadow-lg">
            <div class="modal-header bg-gradient-primary text-white border-0">
                <h5 class="modal-title" id="otpModalLabel">
                    <i class="fas fa-shield-alt mr-2"></i>Phone Number Verification
                </h5>
            </div>
            <div class="modal-body p-4">
                <!-- Header -->
                <div class="text-center mb-4">
                    <div class="w-20 h-20 bg-gradient-to-r from-blue-500 to-purple-600 rounded-full flex items-center justify-center mx-auto mb-3" style="width: 80px; height: 80px;">
                        <i class="fas fa-mobile-alt text-3xl text-white"></i>
                    </div>
                    <h6 class="font-weight-bold text-gray-800">Verification Code Sent</h6>
                    <p class="text-muted mb-0">Enter the 6-digit code sent to</p>
                    <p class="font-weight-bold text-primary" id="otp_phone_display"></p>
                </div>

                <!-- OTP Input -->
                <form id="otpVerificationForm">
                    <div class="row justify-content-center mb-4">
                        <div class="col-md-8">
                            <div class="otp-input-container d-flex justify-content-between">
                                <input type="text" class="otp-input form-control text-center" maxlength="1" id="otp1" tabindex="1">
                                <input type="text" class="otp-input form-control text-center" maxlength="1" id="otp2" tabindex="2">
                                <input type="text" class="otp-input form-control text-center" maxlength="1" id="otp3" tabindex="3">
                                <input type="text" class="otp-input form-control text-center" maxlength="1" id="otp4" tabindex="4">
                                <input type="text" class="otp-input form-control text-center" maxlength="1" id="otp5" tabindex="5">
                                <input type="text" class="otp-input form-control text-center" maxlength="1" id="otp6" tabindex="6">
                            </div>
                            <small class="text-muted d-block text-center mt-2">Enter the 6-digit verification code</small>
                        </div>
                    </div>

                    <!-- Timer and Resend -->
                    <div class="text-center mb-4">
                        <div id="otp_timer_container" style="display: none;">
                            <p class="text-muted mb-2">
                                <i class="fas fa-clock mr-1"></i>
                                Code expires in: <span id="otp_timer" class="font-weight-bold text-warning">05:00</span>
                            </p>
                        </div>
                        <div id="otp_resend_container" style="display: none;">
                            <p class="text-muted mb-2">Didn't receive the code?</p>
                            <button type="button" class="btn btn-link p-0 font-weight-bold" id="resend_otp_btn">
                                <i class="fas fa-redo mr-1"></i>Resend Code
                            </button>
                        </div>
                    </div>

                    <!-- Error/Success Messages -->
                    <div id="otp_message_container" style="display: none;">
                        <div class="alert" id="otp_message" role="alert"></div>
                    </div>

                    <!-- Buttons -->
                    <div class="row">
                        <div class="col-6">
                            <button type="button" class="btn btn-secondary btn-block" id="cancel_otp_btn">
                                <i class="fas fa-times mr-1"></i>Cancel
                            </button>
                        </div>
                        <div class="col-6">
                            <button type="submit" class="btn btn-primary btn-block" id="verify_otp_btn">
                                <i class="fas fa-check mr-1"></i>Verify
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- OTP Styles -->
<style>
/* Ensure modal is hidden by default */
#otpModal {
    display: none !important;
}

#otpModal.show {
    display: block !important;
}

.otp-input-container {
    gap: 10px;
}

.otp-input {
    width: 50px !important;
    height: 50px;
    font-size: 20px;
    font-weight: bold;
    border: 2px solid #dee2e6;
    border-radius: 8px;
    transition: all 0.3s ease;
}

.otp-input:focus {
    border-color: #007bff;
    box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25);
    outline: none;
}

.otp-input.is-valid {
    border-color: #28a745;
}

.otp-input.is-invalid {
    border-color: #dc3545;
}

.bg-gradient-primary {
    background: linear-gradient(87deg, #5e72e4 0, #825ee4 100%);
}

.modal-content {
    border-radius: 15px;
}

@media (max-width: 576px) {
    .otp-input {
        width: 40px !important;
        height: 40px;
        font-size: 16px;
    }
    
    .otp-input-container {
        gap: 5px;
    }
}
</style>

<!-- OTP JavaScript -->
<script>
class OTPVerification {
    constructor() {
        this.phone = '';
        this.purpose = '';
        this.timerInterval = null;
        this.timeLeft = 300; // 5 minutes in seconds
        this.maxAttempts = 3;
        this.currentAttempts = 0;
        
        this.initializeEventListeners();
    }
    
    initializeEventListeners() {
        // OTP input handling
        const otpInputs = document.querySelectorAll('.otp-input');
        otpInputs.forEach((input, index) => {
            input.addEventListener('input', (e) => this.handleOTPInput(e, index));
            input.addEventListener('keydown', (e) => this.handleOTPKeydown(e, index));
            input.addEventListener('paste', (e) => this.handleOTPPaste(e));
        });
        
        // Form submission
        document.getElementById('otpVerificationForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.verifyOTP();
        });
        
        // Resend button
        document.getElementById('resend_otp_btn').addEventListener('click', () => {
            if (typeof window.customOTPResend === 'function') {
                window.customOTPResend(this.phone, this.purpose);
            } else {
                this.resendOTP();
            }
        });
        
        // Cancel button
        document.getElementById('cancel_otp_btn').addEventListener('click', () => {
            this.closeModal();
        });
        
        // Modal close event
        $('#otpModal').on('hidden.bs.modal', () => {
            this.resetForm();
        });
    }
    
    showModal(phone, purpose) {
        this.phone = phone;
        this.purpose = purpose;
        this.resetForm();
        
        document.getElementById('otp_phone_display').textContent = this.formatPhoneForDisplay(phone);
        $('#otpModal').modal('show');
        
        // Focus first input
        setTimeout(() => {
            document.getElementById('otp1').focus();
        }, 300);
        
        this.startTimer();
    }
    
    closeModal() {
        $('#otpModal').modal('hide');
    }
    
    resetForm() {
        // Clear inputs
        document.querySelectorAll('.otp-input').forEach(input => {
            input.value = '';
            input.classList.remove('is-valid', 'is-invalid');
        });
        
        // Reset state
        this.currentAttempts = 0;
        this.timeLeft = 300;
        this.clearTimer();
        
        // Hide messages
        document.getElementById('otp_message_container').style.display = 'none';
        document.getElementById('otp_timer_container').style.display = 'block';
        document.getElementById('otp_resend_container').style.display = 'none';
        
        // Reset buttons
        document.getElementById('verify_otp_btn').disabled = false;
        document.getElementById('resend_otp_btn').disabled = false;
    }
    
    handleOTPInput(event, index) {
        const input = event.target;
        const value = input.value;
        
        // Only allow digits
        if (!/^\d$/.test(value)) {
            input.value = '';
            return;
        }
        
        input.classList.add('is-valid');
        
        // Move to next input
        if (value && index < 5) {
            document.getElementById(`otp${index + 2}`).focus();
        }
        
        // Auto-verify if all fields filled
        if (this.getAllOTPValues().length === 6) {
            setTimeout(() => this.verifyOTP(), 500);
        }
    }
    
    handleOTPKeydown(event, index) {
        // Handle backspace
        if (event.key === 'Backspace' && !event.target.value && index > 0) {
            document.getElementById(`otp${index}`).focus();
        }
        
        // Handle arrow keys
        if (event.key === 'ArrowLeft' && index > 0) {
            event.preventDefault();
            document.getElementById(`otp${index}`).focus();
        }
        
        if (event.key === 'ArrowRight' && index < 5) {
            event.preventDefault();
            document.getElementById(`otp${index + 2}`).focus();
        }
    }
    
    handleOTPPaste(event) {
        event.preventDefault();
        const pasteData = event.clipboardData.getData('text');
        const otpCode = pasteData.replace(/\D/g, '').substring(0, 6);
        
        if (otpCode.length === 6) {
            for (let i = 0; i < 6; i++) {
                const input = document.getElementById(`otp${i + 1}`);
                input.value = otpCode[i];
                input.classList.add('is-valid');
            }
            
            setTimeout(() => this.verifyOTP(), 500);
        }
    }
    
    getAllOTPValues() {
        let otpCode = '';
        for (let i = 1; i <= 6; i++) {
            otpCode += document.getElementById(`otp${i}`).value;
        }
        return otpCode;
    }
    
    async verifyOTP() {
        const otpCode = this.getAllOTPValues();
        
        if (otpCode.length !== 6) {
            this.showMessage('Please enter all 6 digits', 'error');
            return;
        }
        
        // Disable verify button
        const verifyBtn = document.getElementById('verify_otp_btn');
        const originalText = verifyBtn.innerHTML;
        verifyBtn.disabled = true;
        verifyBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-1"></i>Verifying...';
        let phone = $('#check_phone').val();
        let purpose = 'patient_registration';
        try {
            const response = await fetch('<?php echo base_url("frontend/verify_otp"); ?>', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    phone: phone,
                    otp_code: otpCode,
                    purpose: purpose
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showMessage('Phone number verified successfully!', 'success');
                
                // Mark inputs as valid
                document.querySelectorAll('.otp-input').forEach(input => {
                    input.classList.add('is-valid');
                    input.classList.remove('is-invalid');
                });
                
                // Trigger success callback
                if (typeof window.otpVerificationSuccess === 'function') {
                    window.otpVerificationSuccess(this.phone, this.purpose);
                }
                
                setTimeout(() => {
                    this.closeModal();
                }, 1500);
                
            } else {
                this.currentAttempts++;
                this.showMessage(result.message, 'error');
                
                // Mark inputs as invalid
                document.querySelectorAll('.otp-input').forEach(input => {
                    input.classList.add('is-invalid');
                    input.classList.remove('is-valid');
                    input.value = '';
                });
                
                // Focus first input
                document.getElementById('otp1').focus();
                
                // Disable if max attempts reached
                if (this.currentAttempts >= this.maxAttempts) {
                    verifyBtn.disabled = true;
                    verifyBtn.innerHTML = 'Max attempts reached';
                    this.showResendOption();
                }
            }
            
        } catch (error) {
            console.error('OTP verification error:', error);
            this.showMessage('Network error. Please try again.', 'error');
        } finally {
            if (verifyBtn.disabled && this.currentAttempts < this.maxAttempts) {
                verifyBtn.disabled = false;
                verifyBtn.innerHTML = originalText;
            }
        }
    }
    
    async resendOTP() {
        const resendBtn = document.getElementById('resend_otp_btn');
        const originalText = resendBtn.innerHTML;
        resendBtn.disabled = true;
        resendBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-1"></i>Sending...';
        
        try {
            const response = await fetch('<?php echo base_url("frontend/resend_otp"); ?>', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    phone: this.phone,
                    purpose: this.purpose
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showMessage('New verification code sent!', 'success');
                this.resetForm();
                this.startTimer();
                
                // Focus first input
                setTimeout(() => {
                    document.getElementById('otp1').focus();
                }, 300);
                
            } else {
                this.showMessage(result.message, 'error');
            }
            
        } catch (error) {
            console.error('OTP resend error:', error);
            this.showMessage('Failed to resend code. Please try again.', 'error');
        } finally {
            resendBtn.disabled = false;
            resendBtn.innerHTML = originalText;
        }
    }
    
    startTimer() {
        this.clearTimer();
        this.timeLeft = 300; // 5 minutes
        
        document.getElementById('otp_timer_container').style.display = 'block';
        document.getElementById('otp_resend_container').style.display = 'none';
        
        this.timerInterval = setInterval(() => {
            this.timeLeft--;
            
            const minutes = Math.floor(this.timeLeft / 60);
            const seconds = this.timeLeft % 60;
            const timeString = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
            
            document.getElementById('otp_timer').textContent = timeString;
            
            if (this.timeLeft <= 0) {
                this.clearTimer();
                this.showResendOption();
                this.showMessage('Verification code has expired. Please request a new one.', 'warning');
            }
        }, 1000);
    }
    
    clearTimer() {
        if (this.timerInterval) {
            clearInterval(this.timerInterval);
            this.timerInterval = null;
        }
    }
    
    showResendOption() {
        document.getElementById('otp_timer_container').style.display = 'none';
        document.getElementById('otp_resend_container').style.display = 'block';
    }
    
    showMessage(message, type) {
        const container = document.getElementById('otp_message_container');
        const messageEl = document.getElementById('otp_message');
        
        // Remove existing classes
        messageEl.className = 'alert';
        
        // Add appropriate class
        switch (type) {
            case 'success':
                messageEl.classList.add('alert-success');
                break;
            case 'error':
                messageEl.classList.add('alert-danger');
                break;
            case 'warning':
                messageEl.classList.add('alert-warning');
                break;
            default:
                messageEl.classList.add('alert-info');
        }
        
        messageEl.textContent = message;
        container.style.display = 'block';
        
        // Auto-hide success messages
        if (type === 'success') {
            setTimeout(() => {
                container.style.display = 'none';
            }, 3000);
        }
    }
    
    formatPhoneForDisplay(phone) {
        // Format phone number for display (hide middle digits)
        if (phone.length >= 10) {
            const start = phone.substring(0, 3);
            const end = phone.substring(phone.length - 3);
            return `${start}****${end}`;
        }
        return phone;
    }
}

// Initialize OTP verification when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    // Ensure modal is hidden on page load
    const otpModal = document.getElementById('otpModal');
    if (otpModal) {
        otpModal.style.display = 'none';
        otpModal.classList.remove('show');
        otpModal.setAttribute('aria-hidden', 'true');
        
        // Remove modal backdrop if it exists
        const backdrop = document.querySelector('.modal-backdrop');
        if (backdrop) {
            backdrop.remove();
        }
    }
    
    // jQuery-based initialization (if jQuery is available)
    if (typeof jQuery !== 'undefined') {
        jQuery(document).ready(function($) {
            // Ensure modal is hidden
            $('#otpModal').modal('hide');
            $('#otpModal').removeClass('show');
            $('#otpModal').css('display', 'none');
            
            // Remove any backdrop
            $('.modal-backdrop').remove();
        });
    }
    
    // Initialize OTP verification
    window.otpVerification = new OTPVerification();
});

// Helper function to show OTP modal
function showOTPVerification(phone, purpose) {
    if (window.otpVerification) {
        window.otpVerification.showModal(phone, purpose);
    }
}
</script>