<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $settings->title; ?> - Patient Registration</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        'primary': '#667eea',
                        'secondary': '#764ba2',
                    }
                }
            }
        }
    </script>
    <style>
        .gradient-bg {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        .step {
            display: none;
            animation: fadeIn 0.5s ease-in;
        }
        .step.active {
            display: block;
        }
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        .pulse-animation {
            animation: pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite;
        }
        .hospital-card {
            transition: all 0.3s ease;
        }
        .hospital-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
        }
        .hospital-card.selected {
            border-color: #667eea !important;
            background-color: #eff6ff !important;
        }
        
        /* Additional custom styles for modal animations */
        .modal-overlay {
            backdrop-filter: blur(2px);
        }
    </style>
</head>
<body class="font-sans bg-gray-50">
    <!-- Navigation -->
    <nav class="backdrop-blur-md shadow-lg fixed w-full top-0 z-50">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center h-16">
                <!-- Logo -->
                <div class="flex items-center">
                    <a href="<?php echo base_url('frontend/public_portal'); ?>" class="flex items-center">
                        <div class="w-8 h-8 bg-gradient-to-r from-primary to-secondary rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-hospital text-white text-sm"></i>
                        </div>
                        <span class="font-bold text-xl text-white"><?php echo $settings->title; ?></span>
                    </a>
                </div>
                
                <!-- Desktop Navigation -->
                <div class="hidden md:flex items-center space-x-8">
                    <a href="<?php echo base_url('frontend/public_portal'); ?>" class="text-white hover:text-blue-200 transition-colors duration-200 font-medium">
                        <i class="fas fa-home mr-1"></i>Home
                    </a>
                    <a href="<?php echo base_url('frontend/doctor_registration'); ?>" class="text-white hover:text-blue-200 transition-colors duration-200 font-medium">
                        <i class="fas fa-user-md mr-1"></i>Doctor Registration
                    </a>
                    <a href="<?php echo base_url('frontend/appointment_request'); ?>" class="text-white hover:text-blue-200 transition-colors duration-200 font-medium">
                        <i class="fas fa-calendar-plus mr-1"></i>Book Appointment
                    </a>
                    <a href="<?php echo base_url('auth/login'); ?>" class="bg-gradient-to-r from-primary to-secondary text-white px-6 py-2 rounded-lg hover:from-primary-600 hover:to-secondary-600 transition-all duration-200 font-medium">
                        <i class="fas fa-sign-in-alt mr-1"></i>Login
                    </a>
                </div>
                
                <!-- Mobile Menu Button -->
                <div class="md:hidden">
                    <button id="mobile-menu-button" class="text-white hover:text-blue-200 focus:outline-none">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                </div>
            </div>
        </div>
        
        <!-- Mobile Navigation -->
        <div id="mobile-menu" class="hidden md:hidden bg-white border-t border-gray-200">
            <div class="px-2 pt-2 pb-3 space-y-1">
                <a href="<?php echo base_url('frontend/public_portal'); ?>" class="block px-3 py-2 text-gray-700 hover:text-primary hover:bg-gray-50 rounded-md font-medium">
                    <i class="fas fa-home mr-2"></i>Home
                </a>
                <a href="<?php echo base_url('frontend/doctor_registration'); ?>" class="block px-3 py-2 text-gray-700 hover:text-primary hover:bg-gray-50 rounded-md font-medium">
                    <i class="fas fa-user-md mr-2"></i>Doctor Registration
                </a>
                <a href="<?php echo base_url('frontend/appointment_request'); ?>" class="block px-3 py-2 text-gray-700 hover:text-primary hover:bg-gray-50 rounded-md font-medium">
                    <i class="fas fa-calendar-plus mr-2"></i>Book Appointment
                </a>
                <a href="<?php echo base_url('auth/login'); ?>" class="block px-3 py-2 bg-gradient-to-r from-primary to-secondary text-white rounded-md font-medium">
                    <i class="fas fa-sign-in-alt mr-2"></i>Login
                </a>
            </div>
        </div>
    </nav>

    <section class="gradient-bg min-h-screen py-12 pt-32">
        <div class="container mx-auto px-4">
            <div class="max-w-4xl mx-auto">
                <!-- Header -->
                <div class="text-center text-white mb-8">
                    <h1 class="text-4xl md:text-5xl font-bold mb-4">Patient Registration</h1>
                    <p class="text-xl text-blue-100 mb-6">Join our healthcare network and access quality medical services</p>

                </div>
                
                <!-- Main Card -->
                <div class="bg-white rounded-3xl shadow-2xl overflow-hidden">
                    <div class="p-8 md:p-12">
                        <!-- Step Indicator -->
                        <div class="flex justify-center items-center mb-8">
                            <div class="flex items-center space-x-8">
                                <div class="flex items-center space-x-2">
                                    <div class="step-circle w-8 h-8 bg-primary text-white rounded-full flex items-center justify-center text-sm font-medium" id="step-indicator-1">1</div>
                                    <span class="text-sm font-medium text-gray-600">Check</span>
                                </div>
                                <div class="w-16 h-1 bg-gray-200 rounded-full">
                                    <div class="progress-bar w-0 h-full bg-primary rounded-full transition-all duration-500" id="progress-1"></div>
                                </div>
                                <div class="flex items-center space-x-2">
                                    <div class="step-circle w-8 h-8 bg-gray-200 text-gray-400 rounded-full flex items-center justify-center text-sm font-medium" id="step-indicator-2">2</div>
                                    <span class="text-sm font-medium text-gray-400">Hospital</span>
                                </div>
                                <div class="w-16 h-1 bg-gray-200 rounded-full">
                                    <div class="progress-bar w-0 h-full bg-primary rounded-full transition-all duration-500" id="progress-2"></div>
                                </div>
                                <div class="flex items-center space-x-2">
                                    <div class="step-circle w-8 h-8 bg-gray-200 text-gray-400 rounded-full flex items-center justify-center text-sm font-medium" id="step-indicator-3">3</div>
                                    <span class="text-sm font-medium text-gray-400">Registration</span>
                                </div>
                            </div>
                        </div>

                        <!-- Step 1: Mobile Number Check -->
                        <div class="step active" id="step-1">
                            <div class="text-center mb-8">
                                <div class="w-20 h-20 bg-gradient-to-r from-blue-500 to-purple-600 rounded-full flex items-center justify-center mx-auto mb-6">
                                    <i class="fas fa-mobile-alt text-3xl text-white"></i>
                                </div>
                                <h3 class="text-2xl font-bold text-gray-800 mb-3">Mobile Number Verification</h3>
                                <p class="text-gray-600">Let's check if you're already registered with your mobile number</p>
                            </div>
                            
                            <form id="checkPatientForm" class="space-y-6">
                                <div class="max-w-md mx-auto">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Mobile Number <span class="text-red-500">*</span></label>
                                        <input type="tel" id="check_phone" placeholder="Enter your mobile number" required
                                               class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300 text-center text-lg">
                                        <small class="text-gray-500 text-sm mt-1 block">Enter your 10-digit mobile number</small>
                                    </div>
                                </div>
                                
                                <div class="flex justify-center space-x-4">
                                    <button type="button" onclick="checkExistingPatient()" 
                                            class="px-8 py-3 bg-gradient-to-r from-blue-500 to-purple-600 text-white font-semibold rounded-xl hover:from-blue-600 hover:to-purple-700 transform hover:scale-105 transition-all duration-300 shadow-lg">
                                        <i class="fas fa-search mr-2"></i>Check Mobile Number
                                    </button>
                                </div>
                                
                                <!-- Phone validation result -->
                                <div id="phoneCheckResult" style="display: none;" class="mt-6 p-4 rounded-xl border max-w-md mx-auto">
                                    <!-- Results will be shown here -->
                                </div>
                            </form>
                        </div>

                        <!-- Step 2: Hospital Selection -->
                        <div class="step" id="step-2">
                            <div class="text-center mb-8">
                                <div class="w-20 h-20 bg-gradient-to-r from-indigo-500 to-blue-600 rounded-full flex items-center justify-center mx-auto mb-6">
                                    <i class="fas fa-hospital text-3xl text-white"></i>
                                </div>
                                <h3 class="text-2xl font-bold text-gray-800 mb-3">Select Hospital</h3>
                                <p class="text-gray-600">Choose the hospital where you want to register</p>
                                <?php if (isset($hospitals)): ?>
                                    <p class="text-sm text-blue-600">Available hospitals: <?php echo count($hospitals); ?></p>
                                <?php else: ?>
                                    <p class="text-sm text-red-600">No hospitals variable found</p>
                                <?php endif; ?>
                            </div>
                            
                            <form id="hospitalSelectionForm" class="space-y-6">
                                <div class="max-w-2xl mx-auto">
                                    <div class="grid gap-4">
                                        <?php 
                                        // Debug output
                                        echo "<!-- Debug: Hospitals count: " . (isset($hospitals) ? count($hospitals) : 0) . " -->";
                                        if (isset($hospitals) && !empty($hospitals)): 
                                        foreach ($hospitals as $hospital): ?>
                                            <div class="hospital-card border-2 border-gray-200 rounded-xl p-6 cursor-pointer hover:border-primary hover:bg-blue-50 transition-all duration-300" 
                                                 data-hospital-id="<?php echo $hospital->id; ?>" 
                                                 data-hospital-name="<?php echo htmlspecialchars($hospital->name); ?>">
                                                <div class="flex items-center">
                                                    <div class="w-12 h-12 bg-gradient-to-r from-blue-500 to-purple-600 rounded-full flex items-center justify-center mr-4">
                                                        <i class="fas fa-hospital text-white"></i>
                                                    </div>
                                                    <div class="flex-1">
                                                        <h4 class="text-lg font-semibold text-gray-800"><?php echo $hospital->name; ?></h4>
                                                        <?php if (!empty($hospital->address)): ?>
                                                            <p class="text-gray-600 text-sm"><?php echo $hospital->address; ?></p>
                                                        <?php endif; ?>
                                                        <?php if (!empty($hospital->phone)): ?>
                                                            <p class="text-gray-500 text-sm">
                                                                <i class="fas fa-phone mr-1"></i><?php echo $hospital->phone; ?>
                                                            </p>
                                                        <?php endif; ?>
                                                    </div>
                                                    <div class="w-6 h-6 border-2 border-gray-300 rounded-full flex items-center justify-center hospital-radio">
                                                        <div class="w-3 h-3 bg-primary rounded-full hidden"></div>
                                                    </div>
                                                </div>
                                            </div>
                                        <?php endforeach; 
                                        else: ?>
                                            <div class="text-center py-8">
                                                <i class="fas fa-hospital text-gray-400 text-4xl mb-4"></i>
                                                <p class="text-gray-500">No hospitals available for registration.</p>
                                                <p class="text-sm text-gray-400">Please contact support for assistance.</p>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                
                                <div class="flex flex-col sm:flex-row gap-4 justify-between pt-6">
                                    <button type="button" onclick="goToStep(1)" 
                                            class="px-8 py-3 border-2 border-gray-300 text-gray-700 font-semibold rounded-xl hover:border-primary hover:text-primary transition-all duration-300">
                                        <i class="fas fa-arrow-left mr-2"></i>Back
                                    </button>
                                    <button type="button" id="continueToRegistration" disabled
                                            class="px-8 py-3 bg-gradient-to-r from-indigo-500 to-blue-600 text-white font-semibold rounded-xl hover:from-indigo-600 hover:to-blue-700 transform hover:scale-105 transition-all duration-300 shadow-lg disabled:opacity-50 disabled:cursor-not-allowed disabled:transform-none">
                                        <i class="fas fa-arrow-right mr-2"></i>Continue to Registration
                                    </button>
                                </div>
                            </form>
                        </div>

                        <!-- Step 3: Registration Form -->
                        <div class="step" id="step-3">
                            <div class="text-center mb-8">
                                <div class="w-20 h-20 bg-gradient-to-r from-purple-500 to-pink-600 rounded-full flex items-center justify-center mx-auto mb-6">
                                    <i class="fas fa-user-plus text-3xl text-white"></i>
                                </div>
                                <h3 class="text-2xl font-bold text-gray-800 mb-3">Complete Registration</h3>
                                <div id="selected-hospital-info" class="bg-blue-50 border border-blue-200 rounded-xl p-4 max-w-md mx-auto mb-4">
                                    <!-- Selected hospital info will be shown here -->
                                </div>
                                <p class="text-gray-600">Please fill in your details to complete registration</p>
                            </div>
                            
                            <form id="registrationForm" class="space-y-6">
                                <input type="hidden" id="selected_hospital_id" name="hospital_id" value="">
                                <input type="hidden" id="registration_phone" name="phone" value="">
                                
                                <div class="grid md:grid-cols-2 gap-6">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Full Name <span class="text-red-500">*</span></label>
                                        <input type="text" name="name" required placeholder="Enter your full name"
                                               class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300">
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Phone Number <span class="text-red-500">*</span></label>
                                        <input type="tel" name="phone" id="registration_phone_visible" required placeholder="Enter your phone number"
                                               class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300">
                                        <div id="phone_validation_message" class="mt-2 text-sm" style="display: none;"></div>
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Email Address</label>
                                        <input type="email" name="email" placeholder="Enter your email (optional)"
                                               class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300">
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Age <span class="text-red-500">*</span></label>
                                        <input type="number" name="age" required placeholder="Enter your age" min="1" max="120"
                                               class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300">
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Gender <span class="text-red-500">*</span></label>
                                        <select name="sex" required class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300">
                                            <option value="">Select Gender</option>
                                            <option value="Male">Male</option>
                                            <option value="Female">Female</option>
                                            <option value="Other">Other</option>
                                        </select>
                                    </div>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Address <span class="text-red-500">*</span></label>
                                    <textarea name="address" required placeholder="Enter your complete address" rows="3"
                                              class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300"></textarea>
                                </div>
                                
                                <div class="flex flex-col sm:flex-row gap-4 justify-between pt-6">
                                    <button type="button" onclick="goToStep(2)" 
                                            class="px-8 py-3 border-2 border-gray-300 text-gray-700 font-semibold rounded-xl hover:border-primary hover:text-primary transition-all duration-300">
                                        <i class="fas fa-arrow-left mr-2"></i>Back to Hospital Selection
                                    </button>
                                    <button type="submit" 
                                            class="px-8 py-3 bg-gradient-to-r from-purple-500 to-pink-600 text-white font-semibold rounded-xl hover:from-purple-600 hover:to-pink-700 transform hover:scale-105 transition-all duration-300 shadow-lg">
                                        <i class="fas fa-user-plus mr-2"></i>Complete Registration
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <script>
        function goToStep(stepNumber) {
            console.log('🚀 goToStep called with stepNumber:', stepNumber);
            
            // Hide all steps
            $('.step').removeClass('active');
            console.log('📄 Removed active class from all steps');
            
            // Reset all indicators
            for (let i = 1; i <= 3; i++) { // We have 3 steps
                const indicator = document.getElementById(`step-indicator-${i}`);
                
                if (i < stepNumber) {
                    indicator.className = 'step-circle w-8 h-8 bg-green-500 text-white rounded-full flex items-center justify-center text-sm font-medium';
                } else if (i === stepNumber) {
                    indicator.className = 'step-circle w-8 h-8 bg-primary text-white rounded-full flex items-center justify-center text-sm font-medium pulse-animation';
                } else {
                    indicator.className = 'step-circle w-8 h-8 bg-gray-200 text-gray-400 rounded-full flex items-center justify-center text-sm font-medium';
                }
            }
            console.log('🔢 Updated step indicators');
            
            // Update progress bars (we have 2 progress bars between 3 steps)
            for (let i = 1; i <= 2; i++) {
                const progressBar = document.getElementById(`progress-${i}`);
                if (progressBar) {
                    if (i < stepNumber - 1) {
                        progressBar.style.width = '100%'; // Full progress
                    } else {
                        progressBar.style.width = '0%'; // No progress
                    }
                }
            }
            console.log('📊 Updated progress bars');
            
            // Show target step
            $(`#step-${stepNumber}`).addClass('active');
            console.log(`✅ Added active class to #step-${stepNumber}`);
            
            // Verify the step is visible
            const targetStep = $(`#step-${stepNumber}`);
            if (targetStep.length > 0) {
                console.log(`✅ Step ${stepNumber} element found and should be visible`);
            } else {
                console.log(`❌ Step ${stepNumber} element not found!`);
            }
        }

        let phoneVerified = false;
        let verifiedPhone = '';

        // Prevent form submission
        $('#checkPatientForm').on('submit', function(e) {
            e.preventDefault();
            checkExistingPatient();
            return false;
        });



        function checkExistingPatient() {
            const phone = $('#check_phone').val().trim();
            
            if (!phone) {
                Swal.fire('Error', 'Please enter your mobile number', 'error');
                return;
            }

            // Hide any previous results
            $('#phoneCheckResult').hide();
            
            $.ajax({
                url: '<?php echo base_url('frontend/check_existing_patient_by_phone'); ?>',
                method: 'POST',
                data: { phone: phone },
                dataType: 'json',
                success: function(response) {
                    if (response.exists) {
                        if (response.is_patient) {
                            // User exists and is a patient
                            const patient = response.patient;
                            $('#phoneCheckResult').html(`
                                <div class="bg-green-50 border-green-200">
                                    <div class="text-center">
                                        <i class="fas fa-user text-3xl text-green-600 mb-3"></i>
                                        <h4 class="text-xl font-semibold text-green-800 mb-3">Patient Found!</h4>
                                        <div class="space-y-2 text-green-700 mb-4">
                                            <p><strong>Name:</strong> ${patient.name}</p>
                                            <p><strong>Phone:</strong> ${patient.phone}</p>
                                            <p><strong>Hospital:</strong> ${patient.hospital_name}</p>
                                        </div>
                                        <div class="space-y-2">
                                            <button onclick="window.location.href='<?php echo base_url('frontend/appointment_request'); ?>'" 
                                                    class="w-full px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700">
                                                <i class="fas fa-calendar-plus mr-2"></i>Book Appointment
                                            </button>
                                            <button onclick="resetForm()" 
                                                    class="w-full px-4 py-2 bg-gray-300 text-gray-700 rounded-lg hover:bg-gray-400">
                                                Try Different Number
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            `).show();
                        } else {
                            // User exists but is not a patient (doctor, admin, etc.)
                            $('#phoneCheckResult').html(`
                                <div class="bg-red-50 border-red-200">
                                    <div class="text-center">
                                        <i class="fas fa-exclamation-triangle text-3xl text-red-600 mb-3"></i>
                                        <h4 class="text-xl font-semibold text-red-800 mb-3">Phone Number Already in Use</h4>
                                        <p class="text-red-700 mb-4">${response.message}</p>
                                        <button onclick="resetForm()" 
                                                class="w-full px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700">
                                            Try Different Number
                                        </button>
                                    </div>
                                </div>
                            `).show();
                        }
                    } else {
                        // Phone number is available - show verification option
                        $('#phoneCheckResult').html(`
                            <div class="bg-blue-50 border-blue-200">
                                <div class="text-center">
                                    <i class="fas fa-check-circle text-3xl text-blue-600 mb-3"></i>
                                    <h4 class="text-xl font-semibold text-blue-800 mb-3">Phone Number Available!</h4>
                                    <p class="text-blue-700 mb-4">This phone number is available for registration. Please verify ownership to continue.</p>
                                    <button type="button" onclick="startPhoneVerification('${phone}')" 
                                            class="w-full px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                                        <i class="fas fa-shield-alt mr-2"></i>Verify Phone Number
                                    </button>
                                </div>
                            </div>
                        `).show();
                    }
                },
                error: function() {
                    $('#phoneCheckResult').html(`
                        <div class="bg-red-50 border-red-200">
                            <div class="text-center">
                                <i class="fas fa-exclamation-triangle text-3xl text-red-600 mb-3"></i>
                                <h4 class="text-xl font-semibold text-red-800 mb-3">Error</h4>
                                <p class="text-red-700 mb-4">Something went wrong. Please try again.</p>
                                <button onclick="$('#phoneCheckResult').hide()" 
                                        class="w-full px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700">
                                    Try Again
                                </button>
                            </div>
                        </div>
                    `).show();
                }
            });
        }

        // Make function globally accessible
        window.startPhoneVerification = function(phone) {
                            // Don't send OTP immediately - just show the two-step modal
                console.log('📱 Opening patient OTP verification modal for phone check...');
                showPatientOtpModal(phone, 'phone_check');
            
            return false;
        }

        window.showSimpleOTPModal = function(phone) {
            // Check if modal exists
            if ($('#otpModal').length === 0) {
                alert('System error. Please refresh the page and try again.');
                return;
            }
            
            // Show Bootstrap modal with proper positioning
            const modal = document.getElementById('otpModal');
            
            // Ensure modal has proper Bootstrap modal positioning
            modal.style.display = 'block';
            modal.style.position = 'fixed';
            modal.style.top = '0';
            modal.style.left = '0';
            modal.style.width = '100%';
            modal.style.height = '100%';
            modal.style.zIndex = '1050';
            modal.style.outline = '0';
            
            modal.classList.add('show');
            modal.setAttribute('aria-hidden', 'false');
            modal.setAttribute('aria-modal', 'true');
            modal.setAttribute('role', 'dialog');
            
            // Ensure modal dialog is properly centered and sized
            const modalDialog = modal.querySelector('.modal-dialog');
            if (modalDialog) {
                modalDialog.style.position = 'relative';
                modalDialog.style.width = '500px';
                modalDialog.style.maxWidth = '90vw';
                modalDialog.style.margin = '3rem auto';
                modalDialog.style.pointerEvents = 'none';
                modalDialog.style.display = 'flex';
                modalDialog.style.alignItems = 'center';
                modalDialog.style.minHeight = 'calc(100vh - 6rem)';
            }
            
            const modalContent = modal.querySelector('.modal-content');
            if (modalContent) {
                modalContent.style.position = 'relative';
                modalContent.style.pointerEvents = 'auto';
                modalContent.style.backgroundColor = '#fff';
                modalContent.style.border = '1px solid rgba(0,0,0,.2)';
                modalContent.style.borderRadius = '0.3rem';
                modalContent.style.boxShadow = '0 0.25rem 0.5rem rgba(0,0,0,.5)';
            }
            
            // Add Bootstrap backdrop
            if (!document.querySelector('.modal-backdrop')) {
                const backdrop = document.createElement('div');
                backdrop.className = 'modal-backdrop fade show';
                backdrop.style.position = 'fixed';
                backdrop.style.top = '0';
                backdrop.style.left = '0';
                backdrop.style.width = '100vw';
                backdrop.style.height = '100vh';
                backdrop.style.zIndex = '1040';
                backdrop.style.backgroundColor = 'rgba(0,0,0,0.5)';
                document.body.appendChild(backdrop);
            }
            
            // Add modal-open class to body (Bootstrap standard)
            document.body.classList.add('modal-open');
            document.body.style.overflow = 'hidden';
            
            // Set phone display
            const maskedPhone = phone.replace(/(\d{3})\d{4}(\d{3})/, '$1****$2');
            $('#otp_phone_display').text(maskedPhone);
            
            // Show loading message
            $('#otp_message_container').show();
            $('#otp_message').removeClass().addClass('alert alert-info').text('Sending verification code to your phone...');
            
            // Disable inputs initially
            $('.otp-input').prop('disabled', true).val('');
            $('#verify_otp_btn').prop('disabled', true);
            
            // Hide timer and resend initially
            $('#otp_timer_container').hide();
            $('#otp_resend_container').hide();
            
            // Fix button styling
            $('#verify_otp_btn').removeClass().addClass('btn btn-success btn-lg px-4 py-2 font-weight-bold');
            $('#cancel_otp_btn').removeClass().addClass('btn btn-outline-secondary btn-lg px-4 py-2');
            $('#resend_otp_btn').removeClass().addClass('btn btn-outline-primary btn-sm');
        }

        // Function to close Bootstrap modal properly
        window.closeOTPModal = function() {
            const modal = document.getElementById('otpModal');
            const backdrop = document.querySelector('.modal-backdrop');
            
            // Hide modal
            modal.style.display = 'none';
            modal.classList.remove('show');
            modal.setAttribute('aria-hidden', 'true');
            modal.removeAttribute('aria-modal');
            
            // Remove backdrop
            if (backdrop) {
                backdrop.remove();
            }
            
            // Remove modal-open class from body and restore overflow
            document.body.classList.remove('modal-open');
            document.body.style.overflow = 'auto';
        }

        window.sendOTPInBackground = function(phone) {
            
            $.ajax({
                url: '<?php echo base_url('frontend/send_otp'); ?>',
                method: 'POST',
                data: {
                    phone: phone,
                    purpose: 'patient_registration'
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        // Success - enable OTP input
                        $('#otp_message').removeClass().addClass('alert alert-success').text('Verification code sent to your phone. Please enter the 6-digit code below.');
                        
                        // Enable inputs and remove disabled state
                        $('.otp-input').prop('disabled', false).removeClass('disabled');
                        $('#verify_otp_btn').prop('disabled', false);
                        
                        // Start timer
                        $('#otp_timer_container').show();
                        startOTPTimer();
                        
                        // Focus first input
                        setTimeout(() => {
                            $('#otp1').focus();
                            $('#otp1').click();
                        }, 500);
                        
                    } else {
                        // Failed - show error
                        const errorMsg = response.message || 'Failed to send verification code';
                        $('#otp_message').removeClass().addClass('alert alert-danger').text('Failed to send verification code. ' + errorMsg);
                        
                        // Enable inputs anyway so user can try manual entry
                        $('.otp-input').prop('disabled', false).removeClass('disabled');
                        $('#verify_otp_btn').prop('disabled', false);
                        
                        // Show resend option
                        $('#otp_resend_container').show();
                    }
                },
                error: function() {
                    $('#otp_message').removeClass().addClass('alert alert-danger').text('Network error occurred. Please try resend.');
                    $('.otp-input').prop('disabled', false).removeClass('disabled');
                    $('#verify_otp_btn').prop('disabled', false);
                    $('#otp_resend_container').show();
                }
            });
        }

        function startOTPTimer() {
            let timeLeft = 300; // 5 minutes
            
            function updateTimer() {
                const minutes = Math.floor(timeLeft / 60);
                const seconds = timeLeft % 60;
                $('#otp_timer').text(`${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`);
                
                if (timeLeft <= 0) {
                    $('#otp_timer_container').hide();
                    $('#otp_resend_container').show();
                    return;
                }
                
                timeLeft--;
                setTimeout(updateTimer, 1000);
            }
            
            updateTimer();
        }

        function showOTPModalWithMessage(phone, purpose, response) {
            // Always show the modal first
            if (typeof showOTPVerification === 'function') {
                showOTPVerification(phone, purpose);
            } else if (window.otpVerification && typeof window.otpVerification.showModal === 'function') {
                window.otpVerification.showModal(phone, purpose);
            } else {
                // Manual modal show
                $('#otpModal').modal('show');
                $('#otp_phone_display').text(phone.replace(/(\d{3})\d{4}(\d{3})/, '$1****$2'));
            }
            
            // Wait for modal to show, then display message
            setTimeout(() => {
                if (response.success) {
                    // Success - show success message and enable OTP input
                    showOTPModalMessage('OTP sent successfully! Enter the 6-digit code sent to your phone.', 'success');
                    
                    // Enable OTP inputs
                    $('.otp-input').prop('disabled', false);
                    $('#verify_otp_btn').prop('disabled', false);
                    
                    // Focus first input
                    $('#otp1').focus();
                } else {
                    // Failure - show error message but keep modal open
                    showOTPModalMessage(response.message || 'Failed to send OTP. You can try resending.', 'error');
                    
                    // Disable OTP inputs but enable resend
                    $('.otp-input').prop('disabled', true);
                    $('#verify_otp_btn').prop('disabled', true);
                    
                    // Show resend option immediately
                    $('#otp_timer_container').hide();
                    $('#otp_resend_container').show();
                }
            }, 500);
        }

        function showOTPModalMessage(message, type) {
            const container = $('#otp_message_container');
            const messageEl = $('#otp_message');
            
            // Remove existing classes
            messageEl.removeClass('alert-success alert-danger alert-warning alert-info');
            
            // Add appropriate class
            switch (type) {
                case 'success':
                    messageEl.addClass('alert-success');
                    break;
                case 'error':
                    messageEl.addClass('alert-danger');
                    break;
                case 'warning':
                    messageEl.addClass('alert-warning');
                    break;
                default:
                    messageEl.addClass('alert-info');
            }
            
            messageEl.text(message);
            container.show();
        }

        function onPhoneVerified(phone) {
             console.log('✅ onPhoneVerified called with phone:', phone);
             
            phoneVerified = true;
            verifiedPhone = phone;
             isPhoneValid = true; // ✅ IMPORTANT: Mark phone as valid after OTP verification
             
             console.log('📝 Updated variables - phoneVerified:', phoneVerified, 'verifiedPhone:', verifiedPhone, 'isPhoneValid:', isPhoneValid);
            
            // Update the result to show verified status
            $('#phoneCheckResult').html(`
                <div class="bg-green-50 border-green-200">
                    <div class="text-center">
                        <i class="fas fa-check-circle text-3xl text-green-600 mb-3"></i>
                        <h4 class="text-xl font-semibold text-green-800 mb-3">Phone Verified Successfully!</h4>
                        <p class="text-green-700 mb-4">Your phone number ${phone} has been verified. You can now proceed to register.</p>
                        <button onclick="proceedToHospitalSelection()" 
                                class="w-full px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700">
                            <i class="fas fa-arrow-right mr-2"></i>Continue to Hospital Selection
                        </button>
                    </div>
                </div>
            `);
            
            console.log('🎯 Phone verification UI updated, user should see "Continue to Hospital Selection" button');
        }

        function proceedToHospitalSelection() {
            console.log('🏥 proceedToHospitalSelection called - phoneVerified:', phoneVerified, 'verifiedPhone:', verifiedPhone);
            
            if (!phoneVerified) {
                console.log('❌ Phone not verified, showing error');
                Swal.fire('Error', 'Please verify your phone number first', 'error');
                return;
            }
            
            // Set the verified phone in the registration form and lock it
            $('#registration_phone').val(verifiedPhone);
            $('#registration_phone_visible').val(verifiedPhone)
                .prop('readonly', true)
                .prop('disabled', true)
                .addClass('bg-gray-100 cursor-not-allowed border-green-500');
            
            // Add verified badge if not already present
            if (!$('#registration_phone_visible').next('.verified-badge').length) {
                $('#registration_phone_visible').after(
                    '<div class="verified-badge flex items-center mt-2 text-green-600 font-semibold">' +
                    '<i class="fas fa-shield-check mr-2"></i>' +
                    '<span>Phone number verified and locked</span>' +
                    '</div>'
                );
            }
            
            // Disable phone number input handler after verification to prevent changes
            $('#registration_phone_visible').off('input keyup blur');
            
            console.log('📱 Set phone in registration form:', verifiedPhone);
            console.log('➡️ Going to step 2 (hospital selection)');
            
            // Go to hospital selection step
            goToStep(2);
        }

        function resetForm() {
            $('#check_phone').val('');
            $('#phoneCheckResult').hide();
            phoneVerified = false;
            verifiedPhone = '';
        }

                 // Hospital selection functionality
         $(document).on('click', '.hospital-card', function() {
             // Remove selection from all cards
             $('.hospital-card').removeClass('border-primary bg-blue-50 selected');
             $('.hospital-radio div').addClass('hidden');
             $('.hospital-radio').removeClass('border-primary');
             
             // Add selection to clicked card
             $(this).addClass('border-primary bg-blue-50 selected');
             $(this).find('.hospital-radio').addClass('border-primary');
             $(this).find('.hospital-radio div').removeClass('hidden');

             const hospitalId = $(this).data('hospital-id');
             const hospitalName = $(this).data('hospital-name');
             
             $('#selected_hospital_id').val(hospitalId);
             $('#selected-hospital-info').html(`
                 <div class="text-center">
                     <i class="fas fa-hospital text-blue-600 text-xl mb-2"></i>
                     <h5 class="text-lg font-semibold text-gray-800 mb-1">Registering at:</h5>
                     <p class="text-blue-600 font-medium">${hospitalName}</p>
                 </div>
             `);
             $('#continueToRegistration').prop('disabled', false);
         });

        // Continue to registration button click
        $('#continueToRegistration').on('click', function() {
            const phone = $('#check_phone').val();
            $('#registration_phone').val(phone);
            $('#registration_phone_visible').val(phone);
            goToStep(3);
            
            // Trigger phone validation for the pre-filled phone number
            setTimeout(function() {
                $('#registration_phone_visible').trigger('input');
            }, 500);
        });

        // Phone number validation
        let phoneCheckTimeout;
        let isPhoneValid = false;
        
        $('#registration_phone_visible').on('input', function() {
            const phone = $(this).val().trim();
            const messageDiv = $('#phone_validation_message');
            const submitButton = $('#registrationForm button[type="submit"]');
            
            // Clear previous timeout
            clearTimeout(phoneCheckTimeout);
            
            if (phone.length < 10) {
                messageDiv.hide();
                isPhoneValid = false;
                return;
            }
            
            // Show checking message
            messageDiv.html('<i class="fas fa-spinner fa-spin text-blue-500"></i> <span class="text-blue-500">Checking phone number...</span>').show();
            
            // Delay the check to avoid too many requests
            phoneCheckTimeout = setTimeout(function() {
                $.ajax({
                    url: '<?php echo base_url('frontend/check_phone_exists'); ?>',
                    method: 'POST',
                    data: { phone: phone },
                    dataType: 'json',
                    success: function(response) {
                        if (response.exists) {
                            messageDiv.html('<i class="fas fa-times-circle text-red-500"></i> <span class="text-red-500">' + response.message + '</span>').show();
                            submitButton.prop('disabled', true).addClass('opacity-50 cursor-not-allowed');
                            isPhoneValid = false;
                        } else {
                            messageDiv.html('<i class="fas fa-check-circle text-green-500"></i> <span class="text-green-500">Phone number is available</span>').show();
                            submitButton.prop('disabled', false).removeClass('opacity-50 cursor-not-allowed');
                            isPhoneValid = true;
                        }
                    },
                    error: function() {
                        messageDiv.html('<i class="fas fa-exclamation-triangle text-orange-500"></i> <span class="text-orange-500">Could not verify phone number</span>').show();
                        isPhoneValid = false;
                    }
                });
            }, 800); // Wait 800ms after user stops typing
        });

        // Registration form submission
        $('#registrationForm').on('submit', function(e) {
            e.preventDefault();
            
            // Basic client-side validation
            const name = $('input[name="name"]').val().trim();
            const phone = $('#registration_phone_visible').val().trim();
            const age = $('input[name="age"]').val();
            const sex = $('select[name="sex"]').val();
            const address = $('textarea[name="address"]').val().trim();
            
            if (!name) {
                Swal.fire('Validation Error', 'Name is required', 'error');
                return;
            }
            
            if (!phone) {
                Swal.fire('Validation Error', 'Phone number is required', 'error');
                return;
            }
            
            // Check if phone number is verified through OTP (this is the primary validation)
            if (!phoneVerified) {
                Swal.fire('Security Error', 'Please verify your phone number before submitting the form', 'error');
                return;
            }
            
            // Ensure the phone number matches the verified phone (security check)
            if (phone !== verifiedPhone) {
                Swal.fire('Security Error', 'Phone number has been changed after verification. Please verify the new number.', 'error');
                return;
            }
            
            // If phone is verified through OTP, it's automatically valid
            // Only check isPhoneValid if phone was NOT verified through OTP (fallback validation)
            if (!phoneVerified && !isPhoneValid) {
                Swal.fire('Validation Error', 'Please enter a valid and available phone number', 'error');
                return;
            }
            
            if (!age || age < 1 || age > 120) {
                Swal.fire('Validation Error', 'Please enter a valid age (1-120)', 'error');
                return;
            }
            
            if (!sex) {
                Swal.fire('Validation Error', 'Please select gender', 'error');
                return;
            }
            
            if (!address) {
                Swal.fire('Validation Error', 'Address is required', 'error');
                return;
            }
            
            // Sync the visible phone number with the hidden field
            $('#registration_phone').val(phone);
            
            const formData = $(this).serialize();
            
            $.ajax({
                url: '<?php echo base_url('frontend/register_patient'); ?>',
                method: 'POST',
                data: formData,
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        Swal.fire({
                            title: 'Registration Successful!',
                            text: 'You have been registered successfully. Would you like to book an appointment?',
                            icon: 'success',
                            showCancelButton: true,
                            confirmButtonText: 'Book Appointment',
                            cancelButtonText: 'Go to Home'
                        }).then((result) => {
                            if (result.isConfirmed) {
                                window.location.href = '<?php echo base_url('frontend/appointment_request'); ?>';
                            } else {
                                window.location.href = '<?php echo base_url('frontend/public_portal'); ?>';
                            }
                        });
                    } else {
                        Swal.fire('Error', response.message || 'Registration failed. Please try again.', 'error');
                    }
                },
                error: function() {
                    Swal.fire('Error', 'Something went wrong. Please try again.', 'error');
                }
            });
        });

        // Mobile menu functionality
        document.getElementById('mobile-menu-button').addEventListener('click', function() {
            const mobileMenu = document.getElementById('mobile-menu');
            const icon = this.querySelector('i');
            
            if (mobileMenu.classList.contains('hidden')) {
                mobileMenu.classList.remove('hidden');
                icon.classList.remove('fa-bars');
                icon.classList.add('fa-times');
            } else {
                mobileMenu.classList.add('hidden');
                icon.classList.remove('fa-times');
                icon.classList.add('fa-bars');
            }
        });
    </script>

    <!-- Include OTP Verification Modal -->
    <!-- Patient Registration OTP Modal (Pure Tailwind) -->
    <div id="patientOtpModal" class="fixed inset-0 z-50 hidden" aria-labelledby="patientOtpModalLabel" role="dialog" aria-modal="true">
        <!-- Background overlay -->
        <div class="fixed inset-0 bg-gray-500 bg-opacity-75 modal-overlay" onclick="closePatientOtpModal()"></div>
        
        <!-- Modal container - perfectly centered -->
        <div class="fixed inset-0 flex items-center justify-center p-4">
            <!-- Modal panel -->
            <div class="bg-white rounded-lg shadow-xl max-w-lg w-full max-h-screen overflow-y-auto">
                <!-- Modal Header -->
                <div class="bg-gradient-to-r from-purple-600 to-blue-600 px-6 py-4">
                    <div class="flex items-center justify-between">
                        <h3 class="text-lg font-medium text-white" id="patientOtpModalLabel">
                            <i class="fas fa-user-injured mr-2"></i>Patient Phone Verification
                        </h3>
                        <button type="button" class="text-white hover:text-gray-200 focus:outline-none" onclick="closePatientOtpModal()">
                            <span class="sr-only">Close</span>
                            <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                            </svg>
                        </button>
                    </div>
                </div>
                
                <!-- Modal Body -->
                <div class="px-6 py-6">
                    <!-- Step 1: Send OTP -->
                    <div id="patient_otp_step_1" class="text-center">
                        <div class="w-20 h-20 bg-gradient-to-r from-purple-500 to-blue-600 rounded-full flex items-center justify-center mx-auto mb-4">
                            <i class="fas fa-mobile-alt text-3xl text-white"></i>
                        </div>
                        <h4 class="text-lg font-semibold text-gray-800 mb-3">Verify Your Phone Number</h4>
                        <p class="text-gray-600 mb-3">We need to verify your phone number to continue with the registration.</p>
                        <p class="text-lg font-semibold text-purple-600 mb-6" id="patient_phone_display"></p>
                        
                        <button type="button" class="bg-purple-600 hover:bg-purple-700 text-white font-medium py-3 px-8 rounded-lg transition duration-200" id="send_patient_otp_btn">
                            <i class="fas fa-paper-plane mr-2"></i>Send Verification Code
                        </button>
                    </div>

                    <!-- Step 2: Enter OTP -->
                    <div id="patient_otp_step_2" class="text-center hidden">
                        <div class="w-20 h-20 bg-gradient-to-r from-green-500 to-purple-600 rounded-full flex items-center justify-center mx-auto mb-4">
                            <i class="fas fa-sms text-3xl text-white"></i>
                        </div>
                        <h4 class="text-lg font-semibold text-gray-800 mb-2">Verification Code Sent</h4>
                        <p class="text-gray-600 mb-1">Enter the 6-digit code sent to</p>
                        <p class="text-lg font-semibold text-purple-600 mb-6" id="patient_otp_phone_display"></p>

                        <!-- OTP Input -->
                        <form id="patientOtpVerificationForm">
                            <div class="flex justify-center gap-2 mb-4">
                                <input type="text" class="patient-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none transition duration-200" maxlength="1" id="patient_otp1" data-index="1" tabindex="1">
                                <input type="text" class="patient-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none transition duration-200" maxlength="1" id="patient_otp2" data-index="2" tabindex="2">
                                <input type="text" class="patient-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none transition duration-200" maxlength="1" id="patient_otp3" data-index="3" tabindex="3">
                                <input type="text" class="patient-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none transition duration-200" maxlength="1" id="patient_otp4" data-index="4" tabindex="4">
                                <input type="text" class="patient-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none transition duration-200" maxlength="1" id="patient_otp5" data-index="5" tabindex="5">
                                <input type="text" class="patient-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none transition duration-200" maxlength="1" id="patient_otp6" data-index="6" tabindex="6">
                            </div>
                            <p class="text-sm text-gray-500 mb-4">Enter the 6-digit verification code</p>

                            <!-- Timer and Resend -->
                            <div class="mb-4">
                                <div id="patient_otp_timer_container" class="hidden">
                                    <p class="text-gray-600 mb-2">
                                        <i class="fas fa-clock mr-1"></i>
                                        Code expires in: <span id="patient_otp_timer" class="font-semibold text-yellow-600">05:00</span>
                                    </p>
                                </div>
                                <div id="patient_otp_resend_container" class="hidden">
                                    <p class="text-gray-600 mb-2">Didn't receive the code?</p>
                                    <button type="button" class="text-purple-600 hover:text-purple-800 font-semibold focus:outline-none" id="patient_resend_otp_btn">
                                        <i class="fas fa-redo mr-1"></i>Resend Code
                                    </button>
                                </div>
                            </div>

                            <!-- Message Area -->
                            <div id="patient_otp_message" class="hidden p-3 rounded-lg mb-4"></div>

                            <!-- Action Buttons -->
                            <div class="flex justify-center gap-4">
                                <button type="button" class="bg-gray-500 hover:bg-gray-600 text-white font-medium py-2 px-6 rounded-lg transition duration-200" onclick="showPatientOtpStep1()">
                                    <i class="fas fa-arrow-left mr-2"></i>Back
                                </button>
                                <button type="submit" class="bg-green-600 hover:bg-green-700 text-white font-medium py-2 px-6 rounded-lg transition duration-200" id="verify_patient_otp_btn">
                                    <i class="fas fa-check mr-2"></i>Verify Code
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Include Original OTP Verification Modal (for other purposes if needed) -->
    <?php $this->load->view('includes/otp_verification_modal'); ?>
    
    <script>
        // Patient OTP Modal Variables
        let currentPatientPhone = '';
        let currentPatientPurpose = '';
        let patientOtpTimer = null;
        let patientOtpTimeLeft = 300; // 5 minutes

        // Show Patient OTP Modal (Step 1)
        function showPatientOtpModal(phone, purpose) {
            currentPatientPhone = phone;
            currentPatientPurpose = purpose;
            
            $('#patient_phone_display').text(phone);
            $('#patient_otp_phone_display').text(phone);
            
            // Show step 1, hide step 2
            showPatientOtpStep1();
            
            // Show modal with Tailwind classes
            $('#patientOtpModal').removeClass('hidden');
            $('body').addClass('overflow-hidden'); // Prevent background scrolling
        }

        // Close Patient OTP Modal
        function closePatientOtpModal() {
            $('#patientOtpModal').addClass('hidden');
            $('body').removeClass('overflow-hidden'); // Restore background scrolling
            
            // Clear timer if running
            if (patientOtpTimer) {
                clearInterval(patientOtpTimer);
                patientOtpTimer = null;
            }
            
            // Reset form
            resetPatientOtpForm();
        }

        // Show Step 1 (Send OTP button)
        function showPatientOtpStep1() {
            $('#patient_otp_step_1').removeClass('hidden');
            $('#patient_otp_step_2').addClass('hidden');
            $('#patientOtpModalLabel').html('<i class="fas fa-user-injured mr-2"></i>Patient Phone Verification');
        }

        // Show Step 2 (OTP input fields)
        function showPatientOtpStep2() {
            $('#patient_otp_step_1').addClass('hidden');
            $('#patient_otp_step_2').removeClass('hidden');
            $('#patientOtpModalLabel').html('<i class="fas fa-sms mr-2"></i>Enter Verification Code');
            
            // Focus on first OTP input and debug
            setTimeout(() => {
                console.log('🔍 Checking Patient OTP inputs existence:');
                for (let i = 1; i <= 6; i++) {
                    let input = $('#patient_otp' + i);
                    console.log(`patient_otp${i}:`, input.length > 0 ? 'EXISTS' : 'NOT FOUND');
                }
                $('#patient_otp1').focus();
            }, 100);
            
            // Start timer
            startPatientOtpTimer();
        }

        // Reset OTP form
        function resetPatientOtpForm() {
            // Clear OTP inputs
            for (let i = 1; i <= 6; i++) {
                $('#patient_otp' + i).val('');
            }
            
            // Hide message
            $('#patient_otp_message').addClass('hidden');
            
            // Reset timer containers
            $('#patient_otp_timer_container').addClass('hidden');
            $('#patient_otp_resend_container').addClass('hidden');
            
            // Reset button states
            $('#send_patient_otp_btn').prop('disabled', false).html('<i class="fas fa-paper-plane mr-2"></i>Send Verification Code');
            $('#verify_patient_otp_btn').prop('disabled', false).html('<i class="fas fa-check mr-2"></i>Verify Code');
        }

        // Send OTP (triggered by button click in step 1)
        function sendPatientOtp() {
            $('#send_patient_otp_btn').prop('disabled', true).html('<i class="fas fa-spinner fa-spin mr-2"></i>Sending...');
            
            $.ajax({
                url: '<?php echo base_url('frontend/send_otp'); ?>', 
                method: 'POST',
                data: {
                    phone: currentPatientPhone,
                    purpose: 'patient_registration',
                    hospital_id: $('#selected_hospital_id').val() || ''
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        console.log('✅ Patient OTP sent successfully');
                        showPatientOtpStep2();
                    } else {
                        Swal.fire('Error', response.message || 'Failed to send verification code', 'error');
                    }
                },
                error: function() {
                    Swal.fire('Error', 'Network error. Please try again.', 'error');
                },
                complete: function() {
                    $('#send_patient_otp_btn').prop('disabled', false).html('<i class="fas fa-paper-plane mr-2"></i>Send Verification Code');
                }
            });
        }

        // Timer functions
        function startPatientOtpTimer() {
            patientOtpTimeLeft = 300; // 5 minutes
            $('#patient_otp_timer_container').removeClass('hidden');
            $('#patient_otp_resend_container').addClass('hidden');
            
            patientOtpTimer = setInterval(function() {
                patientOtpTimeLeft--;
                updatePatientOtpTimer();
                
                if (patientOtpTimeLeft <= 0) {
                    clearInterval(patientOtpTimer);
                    $('#patient_otp_timer_container').addClass('hidden');
                    $('#patient_otp_resend_container').removeClass('hidden');
                }
            }, 1000);
        }

        function updatePatientOtpTimer() {
            let minutes = Math.floor(patientOtpTimeLeft / 60);
            let seconds = patientOtpTimeLeft % 60;
            $('#patient_otp_timer').text(
                (minutes < 10 ? '0' : '') + minutes + ':' + 
                (seconds < 10 ? '0' : '') + seconds
            );
        }

        // OTP input handling
        function setupPatientOtpInputs() {
            // Handle input navigation for patient OTP inputs using class
            $('.patient-otp-input').on('input', function() {
                if (this.value.length === 1) {
                    let currentIndex = parseInt($(this).data('index'));
                    if (currentIndex < 6) {
                        $('#patient_otp' + (currentIndex + 1)).focus();
                    }
                }
            });

            $('.patient-otp-input').on('keydown', function(e) {
                if (e.key === 'Backspace' && this.value.length === 0) {
                    let currentIndex = parseInt($(this).data('index'));
                    if (currentIndex > 1) {
                        $('#patient_otp' + (currentIndex - 1)).focus();
                    }
                }
            });

            $('.patient-otp-input').on('paste', function(e) {
                e.preventDefault();
                let paste = (e.originalEvent.clipboardData || window.clipboardData).getData('text');
                if (paste.length === 6 && /^\d+$/.test(paste)) {
                    $('.patient-otp-input').each(function(index) {
                        $(this).val(paste.charAt(index));
                    });
                    $('#patient_otp6').focus();
                }
            });
        }

        // Verify OTP
        function verifyPatientOtp() {
            let otp = '';
            console.log('🔍 Collecting Patient OTP from inputs...');
            
            // Collect OTP from inputs using class selector
            $('.patient-otp-input').each(function() {
                let value = $(this).val() || '';
                console.log(`Input ${$(this).attr('id')}: "${value}"`);
                otp += value;
            });
            
            console.log('Complete Patient OTP:', otp, 'Length:', otp.length);
            
            if (otp.length !== 6 || !/^\d{6}$/.test(otp)) {
                console.log('❌ Patient OTP validation failed - Length:', otp.length, 'Value:', otp);
                showPatientOtpMessage('Please enter the complete 6-digit code', 'danger');
                return;
            }
            
            console.log('✅ Patient OTP validation passed, sending to server with otp_code:', otp);
            
            $('#verify_patient_otp_btn').prop('disabled', true).html('<i class="fas fa-spinner fa-spin mr-2"></i>Verifying...');
            
            $.ajax({
                url: '<?php echo base_url('frontend/verify_otp'); ?>',
                method: 'POST',
                data: {
                    phone: currentPatientPhone,
                    otp_code: otp,
                    purpose: 'patient_registration'
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        showPatientOtpMessage('✅ Phone verified successfully!', 'success');
                        setTimeout(() => {
                            closePatientOtpModal();
                            handlePatientOtpSuccess();
                        }, 1500);
                    } else {
                        showPatientOtpMessage(response.message || 'Invalid verification code', 'danger');
                    }
                },
                error: function() {
                    showPatientOtpMessage('Network error. Please try again.', 'danger');
                },
                complete: function() {
                    $('#verify_patient_otp_btn').prop('disabled', false).html('<i class="fas fa-check mr-2"></i>Verify Code');
                }
            });
        }

        // Handle successful OTP verification
        function handlePatientOtpSuccess() {
            if (currentPatientPurpose === 'phone_check') {
                onPhoneVerified(currentPatientPhone);
            } else if (currentPatientPurpose === 'final_registration') {
                submitPatientRegistration();
            }
        }

        // Show message in OTP modal
        function showPatientOtpMessage(message, type) {
            let bgColor = '';
            let textColor = '';
            let borderColor = '';
            
            if (type === 'success') {
                bgColor = 'bg-green-100';
                textColor = 'text-green-800';
                borderColor = 'border-green-400';
            } else if (type === 'danger' || type === 'error') {
                bgColor = 'bg-red-100';
                textColor = 'text-red-800';
                borderColor = 'border-red-400';
            } else if (type === 'warning') {
                bgColor = 'bg-yellow-100';
                textColor = 'text-yellow-800';
                borderColor = 'border-yellow-400';
            } else {
                bgColor = 'bg-blue-100';
                textColor = 'text-blue-800';
                borderColor = 'border-blue-400';
            }
            
            $('#patient_otp_message')
                .removeClass()
                .addClass(`${bgColor} ${textColor} ${borderColor} border p-3 rounded-lg mb-4`)
                .text(message)
                .removeClass('hidden');
                
            setTimeout(() => {
                $('#patient_otp_message').addClass('hidden');
            }, 5000);
        }

        // Initialize Patient OTP Modal
        $(document).ready(function() {
            console.log('🔧 Patient OTP Modal initialized');
            
            setupPatientOtpInputs();
            
            // Send OTP button click
            $('#send_patient_otp_btn').click(function() {
                sendPatientOtp();
            });
            
            // Resend OTP button click
            $('#patient_resend_otp_btn').click(function() {
                sendPatientOtp();
            });
            
            // OTP form submission
            $('#patientOtpVerificationForm').submit(function(e) {
                e.preventDefault();
                verifyPatientOtp();
            });
            
            // Keyboard event handling
            $(document).keydown(function(e) {
                if (e.key === 'Escape' && !$('#patientOtpModal').hasClass('hidden')) {
                    closePatientOtpModal();
                }
            });
        });

        // OTP verification success callback
        window.otpVerificationSuccess = function(phone, purpose) {
            console.log('🔔 OTP Verification Success Callback:', { phone, purpose, phoneVerified });
            
            if (purpose === 'patient_registration') {
                // Check if this is for initial phone verification or final registration
                if (!phoneVerified) {
                    // Initial phone verification - update UI and allow proceeding
                    console.log('📱 Initial phone verification - calling onPhoneVerified');
                    onPhoneVerified(phone);
                } else {
                    // Final registration verification - proceed with registration
                    console.log('📝 Final registration verification - calling submitPatientRegistration');
                    submitPatientRegistration();
                }
            }
        };

        // Simple resend function
        $(document).on('click', '#resend_otp_btn', function() {
            const phone = $('#otp_phone_display').text().replace(/\*/g, ''); // Get phone from display
            
            $(this).prop('disabled', true).html('<i class="fas fa-spinner fa-spin mr-1"></i>Sending...');
            $('#otp_message').removeClass().addClass('alert alert-info').text('Sending new verification code...');
            
            sendOTPInBackground($('#check_phone').val()); // Use original phone number
            
            setTimeout(() => {
                $(this).prop('disabled', false).html('<i class="fas fa-redo mr-1"></i>Resend Code');
            }, 3000);
        });

        // Simple verify function
$(document).on('submit', '#otpVerificationForm', function(e) {
    e.preventDefault();

    let otpCode = '';
    for (let i = 1; i <= 6; i++) {
        otpCode += $('#otp' + i).val();
    }

    if (otpCode.length !== 6) {
        $('#otp_message').removeClass().addClass('alert alert-danger').text('Please enter all 6 digits');
        return;
    }

    let phone = $('#check_phone').val();
    let purpose = 'patient_registration';

    console.log('🔍 Sending OTP data:', {
        phone: phone,
        otp_code: otpCode,
        purpose: purpose
    });

    $('#verify_otp_btn').prop('disabled', true).html('<i class="fas fa-spinner fa-spin mr-1"></i>Verifying...');

    $.ajax({
        url: 'frontend/verify_otp',
        method: 'POST',
        data: {
            phone: phone,
            otp_code: otpCode,
            purpose: purpose
        },
        dataType: 'json',
        success: function(response) {
            console.log('✅ Server response:', response);
            if (response.success) {
                $('#otp_message').removeClass().addClass('alert alert-success').text('✅ Phone verified successfully!');

                // Manually trigger the success callback since this is the simple OTP modal
                setTimeout(() => {
                    $('#otpModal').modal('hide');
                    
                    // Trigger the global success callback
                    if (typeof window.otpVerificationSuccess === 'function') {
                        console.log('🔄 Manually triggering otpVerificationSuccess callback');
                        window.otpVerificationSuccess($('#check_phone').val(), 'patient_registration');
                    } else {
                        console.log('⚠️ otpVerificationSuccess callback not found, calling onPhoneVerified directly');
                        onPhoneVerified($('#check_phone').val());
                    }
                }, 1500);
            } else {
                $('#otp_message').removeClass().addClass('alert alert-danger').text('❌ ' + (response.message || 'Invalid OTP code'));
                $('.otp-input').val('').removeClass('is-valid').addClass('is-invalid');
                $('#otp1').focus();
            }
        },
        error: function(xhr, status, error) {
            console.log('❌ AJAX Error:', error, xhr.responseText);
            $('#otp_message').removeClass().addClass('alert alert-danger').text('❌ Network error. Please try again.');
        },
        complete: function() {
            $('#verify_otp_btn').prop('disabled', false).html('<i class="fas fa-check mr-1"></i>Verify');
        }
    });
});


        // Modify the existing registration form submission
        let registrationFormData = null;
        let isOTPVerified = false;

        // Override the original form submission
        $('#registrationForm').off('submit').on('submit', function(e) {
            e.preventDefault();
            
            // Store form data
            registrationFormData = $(this).serialize();
            
            // Basic client-side validation (keep existing validation)
            const name = $('input[name="name"]').val().trim();
            const phone = $('#registration_phone_visible').val().trim();
            const age = $('input[name="age"]').val();
            const sex = $('select[name="sex"]').val();
            const address = $('textarea[name="address"]').val().trim();
            
            if (!name) {
                Swal.fire('Validation Error', 'Name is required', 'error');
                return;
            }
            
            if (!phone) {
                Swal.fire('Validation Error', 'Phone number is required', 'error');
                return;
            }
            
            // If phone is verified through OTP, it's automatically valid
            // Only check isPhoneValid if phone was NOT verified through OTP (fallback validation)
            if (!phoneVerified && !isPhoneValid) {
                Swal.fire('Validation Error', 'Please enter a valid and available phone number', 'error');
                return;
            }
            
            if (!age || age < 1 || age > 120) {
                Swal.fire('Validation Error', 'Please enter a valid age (1-120)', 'error');
                return;
            }
            
            if (!sex) {
                Swal.fire('Validation Error', 'Please select gender', 'error');
                return;
            }
            
            if (!address) {
                Swal.fire('Validation Error', 'Address is required', 'error');
                return;
            }
            
            // Sync the visible phone number with the hidden field
            $('#registration_phone').val(phone);
            registrationFormData = $(this).serialize();
            
            // Check if OTP is already verified for this phone
            $.ajax({
                url: '<?php echo base_url('frontend/check_verified_otp'); ?>',
                method: 'POST',
                data: {
                    phone: phone,
                    purpose: 'patient_registration',
                    within_minutes: 30
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success && response.verified) {
                        // Already verified, proceed with registration
                        submitPatientRegistration();
                    } else {
                        // Need OTP verification, send OTP
                        sendOTPForRegistration(phone);
                    }
                },
                error: function() {
                    // If check fails, proceed with OTP verification
                    sendOTPForRegistration(phone);
                }
            });
        });

        function sendOTPForRegistration(phone) {
            // Don't send OTP immediately - just show the two-step modal
            console.log('📱 Opening patient OTP verification modal for final registration...');
            showPatientOtpModal(phone, 'final_registration');
        }

        function submitPatientRegistration() {
            if (!registrationFormData) {
                Swal.fire('Error', 'Form data not found. Please try again.', 'error');
                return;
            }

            // Show loading
            Swal.fire({
                title: 'Creating Account',
                text: 'Please wait while we create your account...',
                allowOutsideClick: false,
                allowEscapeKey: false,
                showConfirmButton: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });

            $.ajax({
                url: '<?php echo base_url('frontend/register_patient'); ?>',
                method: 'POST',
                data: registrationFormData,
                dataType: 'json',
                success: function(response) {
                    Swal.close();
                    
                    if (response.success) {
                        // Send SMS in background (don't wait for it)
                        if (response.sms_data) {
                            $.ajax({
                                url: '<?php echo base_url('frontend/send_registration_sms'); ?>',
                                method: 'POST',
                                data: response.sms_data,
                                dataType: 'json',
                                success: function(smsResponse) {
                                    console.log('SMS Status:', smsResponse.message);
                                },
                                error: function() {
                                    console.log('SMS sending failed (non-critical)');
                                }
                            });
                        }
                        
                        // Show success message
                        Swal.fire({
                            title: 'Registration Successful!',
                            html: `
                                <div class="text-center">
                                    <i class="fas fa-check-circle text-green-600 text-4xl mb-4"></i>
                                    <p class="text-lg mb-2">Welcome to our Hospital Management System!</p>
                                    <p class="text-gray-600 mb-4">Your registration has been completed successfully.</p>
                                    <div class="bg-blue-50 p-4 rounded-lg mb-4">
                                        <p class="font-semibold text-blue-800">Your Login Details:</p>
                                        <p class="text-blue-700">Phone: <strong>${response.sms_data ? response.sms_data.phone : 'N/A'}</strong></p>
                                        <p class="text-blue-700">Password: <strong>${response.sms_data ? response.sms_data.password : 'N/A'}</strong></p>
                                        <p class="text-sm text-blue-600 mt-2">📱 SMS with login details has been sent to your phone</p>
                                    </div>
                                    <p class="text-gray-600">What would you like to do next?</p>
                                </div>
                            `,
                            icon: 'success',
                            showCancelButton: true,
                            confirmButtonText: '<i class="fas fa-calendar-plus mr-2"></i>Book Appointment',
                            cancelButtonText: '<i class="fas fa-home mr-2"></i>Go to Home',
                            confirmButtonColor: '#059669',
                            cancelButtonColor: '#6b7280'
                        }).then((result) => {
                            if (result.isConfirmed) {
                                window.location.href = '<?php echo base_url('frontend/appointment_request'); ?>';
                            } else {
                                window.location.href = '<?php echo base_url('frontend/public_portal'); ?>';
                            }
                        });
                    } else {
                        Swal.fire('Error', response.message || 'Registration failed. Please try again.', 'error');
                    }
                },
                error: function(xhr, status, error) {
                    Swal.close();
                    
                    console.error('=== REGISTRATION AJAX ERROR ===');
                    console.error('Status:', status);
                    console.error('Error:', error);
                    console.error('Response Text:', xhr.responseText);
                    console.error('Status Code:', xhr.status);
                    console.error('===========================');
                    
                    let errorMessage = 'Something went wrong. Please try again.';
                    if (xhr.responseText) {
                        try {
                            const errorResponse = JSON.parse(xhr.responseText);
                            errorMessage = errorResponse.message || errorMessage;
                        } catch (e) {
                            // If it's not JSON, show the raw response (might be PHP error)
                            if (xhr.responseText.length < 500) {
                                errorMessage = 'Server Error: ' + xhr.responseText;
                            } else {
                                errorMessage = 'Server Error occurred. Check console for details.';
                            }
                        }
                    }
                    
                    Swal.fire('Error', errorMessage, 'error');
                }
            });
        }
    </script>
</body>
</html> 