<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Book Appointment - <?php echo $settings->title; ?></title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <!-- Tailwind CSS -->
    <script src="https://cdn.tailwindcss.com"></script>
    <!-- SweetAlert2 -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <!-- Flatpickr -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css">
    <script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>
    
    <style>
        .step {
            display: none;
        }
        .step.active {
            display: block;
        }
        
        .doctor-card {
            border-radius: 15px;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .doctor-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 35px rgba(0,0,0,0.1);
        }
        
        .doctor-card.selected {
            border: 3px solid #667eea;
            background: linear-gradient(135deg, #f8f9ff 0%, #e0e7ff 100%);
        }
        
        .time-slot {
            border-radius: 10px;
            padding: 12px;
            margin: 5px;
            cursor: pointer;
            transition: all 0.2s ease;
            border: 2px solid #e5e7eb;
        }
        
        .time-slot:hover {
            background: #e0e7ff;
            border-color: #667eea;
        }
        
        .time-slot.selected {
            background: #667eea;
            color: white;
            border-color: #667eea;
        }
        
        .step-indicator {
            background: #f1f5f9;
            border-radius: 25px;
            padding: 8px 20px;
        }
        
        .step-indicator.active {
            background: #667eea;
            color: white;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
            border-radius: 12px;
            padding: 12px 30px;
            font-weight: 600;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }
        
        .border-left-warning {
            border-left: 4px solid #ffc107 !important;
        }
        
        .border-left-danger {
            border-left: 4px solid #dc3545 !important;
        }
        
        .border-left-success {
            border-left: 4px solid #28a745 !important;
        }
    </style>
</head>
<body class="bg-gray-50">
    <!-- Navigation -->
    <nav class="bg-gradient-to-r from-blue-600 to-purple-600 backdrop-blur-md shadow-lg fixed w-full top-0 z-50">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center h-16">
                <!-- Logo -->
                <div class="flex items-center">
                    <a href="<?php echo base_url('frontend/public_portal'); ?>" class="flex items-center">
                        <div class="w-8 h-8 bg-gradient-to-r from-primary to-secondary rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-hospital text-white text-sm"></i>
                        </div>
                        <span class="font-bold text-xl text-white"><?php echo $settings->title; ?></span>
                    </a>
                </div>
                
                <!-- Navigation Links -->
                <div class="hidden md:flex items-center space-x-6">
                    <a href="<?php echo base_url('frontend/public_portal'); ?>" class="text-white hover:text-blue-200 transition-colors duration-200 font-medium">
                        <i class="fas fa-home mr-2"></i>Home
                    </a>
                    <a href="<?php echo base_url('frontend/patient_registration'); ?>" class="text-white hover:text-blue-200 transition-colors duration-200 font-medium">
                        <i class="fas fa-user-plus mr-2"></i>Register as Patient
                    </a>
                    <a href="<?php echo base_url('auth/login'); ?>" class="bg-gradient-to-r from-primary to-secondary text-white px-6 py-2 rounded-lg hover:from-primary-600 hover:to-secondary-600 transition-all duration-200 font-medium">
                        <i class="fas fa-sign-in-alt mr-1"></i>Login
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Main Content -->
    <div class="container mx-auto px-4 py-8 mt-20">
        <!-- Header -->
        <div class="text-center mb-8">
            <h1 class="text-4xl font-bold text-gray-800 mb-4">Book Appointment as Guest</h1>
            <p class="text-lg text-gray-600 mb-6">Schedule your visit without registration - You can complete registration when you arrive at the hospital</p>
            
            <!-- Step Indicators -->
            <div class="flex justify-center space-x-4 mb-8">
                <div class="step-indicator active" id="step-indicator-1">
                    <span class="text-sm font-semibold">1. Personal Info</span>
                </div>
                <div class="step-indicator" id="step-indicator-2">
                    <span class="text-sm font-semibold">2. Select Doctor</span>
                </div>
                <div class="step-indicator" id="step-indicator-3">
                    <span class="text-sm font-semibold">3. Date & Time</span>
                </div>
                <div class="step-indicator" id="step-indicator-4">
                    <span class="text-sm font-semibold">4. Confirmation</span>
                </div>
            </div>
        </div>

        <!-- Step 1: Personal Information -->
        <div class="step active" id="step-1">
            <div class="max-w-7xl mx-auto">
                <div class="row g-4">
                    <!-- Left Side: Form -->
                    <div class="col-lg-8">
                        <div class="bg-white rounded-xl shadow-lg p-8">
                            <h3 class="text-2xl font-bold text-gray-800 mb-6 text-center">
                                <i class="fas fa-user-circle text-primary mr-3"></i>Personal Information
                            </h3>
                            
                            <form id="guestInfoForm">
                                <div class="row g-4">
                                    <div class="col-md-6">
                                        <label class="form-label fw-bold">Full Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control form-control-lg" id="guest_name" name="guest_name" required>
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label fw-bold">Phone Number <span class="text-danger">*</span></label>
                                        <input type="tel" class="form-control form-control-lg" id="guest_phone" name="guest_phone" required onblur="checkPhoneRegistration()" oninput="handlePhoneInput()">
                                        <small class="text-muted">We'll automatically check if this number is already registered</small>
                                    </div>
                            <div class="col-md-6">
                                <label class="form-label fw-bold">Email Address</label>
                                <input type="email" class="form-control form-control-lg" id="guest_email" name="guest_email">
                                <small class="text-muted">Optional - for appointment confirmations</small>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label fw-bold">Age <span class="text-danger">*</span></label>
                                <input type="number" class="form-control form-control-lg" id="guest_age" name="guest_age" min="1" max="120" required>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label fw-bold">Gender <span class="text-danger">*</span></label>
                                <select class="form-control form-control-lg" id="guest_gender" name="guest_gender" required>
                                    <option value="">Select Gender</option>
                                    <option value="Male">Male</option>
                                    <option value="Female">Female</option>
                                    <option value="Other">Other</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label fw-bold">Emergency Contact</label>
                                <input type="tel" class="form-control form-control-lg" id="guest_emergency_contact" name="guest_emergency_contact">
                            </div>
                            <div class="col-12">
                                <label class="form-label fw-bold">Address</label>
                                <textarea class="form-control" id="guest_address" name="guest_address" rows="3"></textarea>
                            </div>
                            <div class="col-12">
                                <label class="form-label fw-bold">Describe Your Problem/Symptoms</label>
                                <textarea class="form-control" id="guest_symptoms" name="guest_symptoms" rows="3" placeholder="Please describe your health concern or reason for visit"></textarea>
                            </div>
                        </div>
                        
                                <div class="text-center mt-6">
                                    <button type="button" class="btn btn-primary btn-lg" onclick="goToStep(2)">
                                        <i class="fas fa-arrow-right mr-2"></i>Continue to Doctor Selection
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                    
                    <!-- Right Side: Phone Validation Result -->
                    <div class="col-lg-4">
                        <div class="sticky-top" style="top: 20px;">
                            <!-- Default Message -->
                            <div id="phoneDefaultMessage" class="bg-gradient-to-r from-blue-50 to-indigo-50 border border-blue-200 rounded-xl p-6 shadow-sm">
                                <div class="text-center">
                                    <div class="mb-4">
                                        <i class="fas fa-phone-alt text-blue-500 text-4xl"></i>
                                    </div>
                                    <h5 class="font-bold text-gray-800 mb-3">Phone Number Verification</h5>
                                    <p class="text-gray-600 text-sm mb-3">
                                        Enter your phone number and we'll automatically check if it's already registered in our system.
                                    </p>
                                    <div class="flex items-center justify-center text-xs text-gray-500">
                                        <i class="fas fa-shield-alt mr-2"></i>
                                        <span>Secure & Private</span>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Validation Result (Hidden by default) -->
                            <div id="phoneCheckResult" class="mt-2" style="display: none;"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Step 2: Doctor Selection -->
        <div class="step" id="step-2">
            <div class="max-w-6xl mx-auto">
                <div class="bg-white rounded-xl shadow-lg p-8">
                    <h3 class="text-2xl font-bold text-gray-800 mb-6 text-center">
                        <i class="fas fa-user-md text-primary mr-3"></i>Select Doctor & Hospital
                    </h3>
                    
                    <!-- Hospital Selection -->
                    <div class="mb-6">
                        <label class="form-label fw-bold">Select Hospital <span class="text-danger">*</span></label>
                        <select class="form-control form-control-lg" id="hospital_select" onchange="loadDoctorsByHospital()">
                            <option value="">Choose Hospital</option>
                            <?php foreach($hospitals as $hospital): ?>
                                <option value="<?php echo $hospital->id; ?>"><?php echo $hospital->name; ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <!-- Specialty Filter -->
                    <div class="mb-6">
                        <label class="form-label fw-bold">Filter by Specialty (Optional)</label>
                        <select class="form-control form-control-lg" id="specialty_filter" onchange="filterDoctors()">
                            <option value="">All Specialties</option>
                            <?php foreach($specialities as $specialty): ?>
                                <option value="<?php echo $specialty->name; ?>"><?php echo $specialty->name; ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <!-- Doctors List -->
                    <div id="doctors_list" class="row g-4">
                        <div class="col-12 text-center text-muted">
                            <i class="fas fa-hospital-user fa-3x mb-3"></i>
                            <p>Please select a hospital to view available doctors</p>
                        </div>
                    </div>
                    
                    <div class="text-center mt-6">
                        <button type="button" class="btn btn-secondary me-3" onclick="goToStep(1)">
                            <i class="fas fa-arrow-left mr-2"></i>Back
                        </button>
                        <button type="button" class="btn btn-primary" onclick="goToStep(3)" id="continueToDateTime" disabled>
                            <i class="fas fa-arrow-right mr-2"></i>Continue to Date & Time
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Step 3: Date & Time Selection -->
        <div class="step" id="step-3">
            <div class="max-w-4xl mx-auto">
                <div class="bg-white rounded-xl shadow-lg p-8">
                    <h3 class="text-2xl font-bold text-gray-800 mb-6 text-center">
                        <i class="fas fa-calendar-alt text-primary mr-3"></i>Select Date & Time
                    </h3>
                    
                    <div class="row g-6">
                        <div class="col-md-6">
                            <h5 class="mb-4">Selected Doctor</h5>
                            <div id="selected_doctor_info" class="bg-light p-4 rounded-lg">
                                <!-- Doctor info will be populated here -->
                            </div>
                            
                            <h5 class="mb-3 mt-4">Select Date</h5>
                            <input type="text" id="appointment_date" class="form-control form-control-lg" placeholder="Choose date" readonly>
                        </div>
                        
                        <div class="col-md-6">
                            <h5 class="mb-3">Available Time Slots</h5>
                            <div id="time_slots_container">
                                <p class="text-muted text-center">Please select a date to view available times</p>
                            </div>
                        </div>
                    </div>
                    
                    <div class="text-center mt-6">
                        <button type="button" class="btn btn-secondary me-3" onclick="goToStep(2)">
                            <i class="fas fa-arrow-left mr-2"></i>Back
                        </button>
                        <button type="button" class="btn btn-primary" onclick="goToStep(4)" id="continueToConfirmation" disabled>
                            <i class="fas fa-arrow-right mr-2"></i>Continue to Confirmation
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Step 4: Confirmation -->
        <div class="step" id="step-4">
            <div class="max-w-3xl mx-auto">
                <div class="bg-white rounded-xl shadow-lg p-8">
                    <h3 class="text-2xl font-bold text-gray-800 mb-6 text-center">
                        <i class="fas fa-check-circle text-success mr-3"></i>Confirm Your Appointment
                    </h3>
                    
                    <div id="appointment_summary">
                        <!-- Appointment summary will be populated here -->
                    </div>
                    
                    <div class="bg-light p-4 rounded-lg mt-4">
                        <h6 class="fw-bold mb-3">Important Notes:</h6>
                        <ul class="list-unstyled">
                            <li class="mb-2"><i class="fas fa-info-circle text-primary me-2"></i>This is a guest appointment - no payment required now</li>
                            <li class="mb-2"><i class="fas fa-clock text-warning me-2"></i>Please arrive 15 minutes before your appointment</li>
                            <li class="mb-2"><i class="fas fa-id-card text-info me-2"></i>Bring valid ID for patient registration at hospital</li>
                            <li class="mb-0"><i class="fas fa-phone text-success me-2"></i>Hospital will contact you for confirmation</li>
                        </ul>
                    </div>
                    
                    <div class="text-center mt-6">
                        <button type="button" class="btn btn-secondary me-3" onclick="goToStep(3)">
                            <i class="fas fa-arrow-left mr-2"></i>Back
                        </button>
                        <button type="button" class="btn btn-success btn-lg" onclick="submitGuestAppointment()">
                            <i class="fas fa-calendar-check mr-2"></i>Confirm Appointment
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let selectedDoctor = null;
        let selectedDate = null;
        let selectedTime = null;
        let availableDoctors = [];

        function goToStep(step) {
            // Validate current step before proceeding
            if (step > 1 && !validateCurrentStep(step - 1)) {
                return;
            }
            
            // Hide all steps
            document.querySelectorAll('.step').forEach(s => s.classList.remove('active'));
            document.querySelectorAll('.step-indicator').forEach(s => s.classList.remove('active'));
            
            // Show target step
            document.getElementById('step-' + step).classList.add('active');
            document.getElementById('step-indicator-' + step).classList.add('active');
            
            // Update step-specific content
            if (step === 4) {
                updateAppointmentSummary();
            }
        }

        function validateCurrentStep(step) {
            switch(step) {
                case 1:
                    const name = document.getElementById('guest_name').value.trim();
                    const phone = document.getElementById('guest_phone').value.trim();
                    const age = document.getElementById('guest_age').value;
                    const gender = document.getElementById('guest_gender').value;
                    
                    if (!name || !phone || !age || !gender) {
                        Swal.fire('Error', 'Please fill in all required fields', 'error');
                        return false;
                    }
                    return true;
                    
                case 2:
                    if (!selectedDoctor) {
                        Swal.fire('Error', 'Please select a doctor', 'error');
                        return false;
                    }
                    return true;
                    
                case 3:
                    if (!selectedDate || !selectedTime) {
                        Swal.fire('Error', 'Please select date and time', 'error');
                        return false;
                    }
                    return true;
            }
            return true;
        }

        function loadDoctorsByHospital() {
            const hospitalId = document.getElementById('hospital_select').value;
            if (!hospitalId) {
                document.getElementById('doctors_list').innerHTML = '<div class="col-12 text-center text-muted"><i class="fas fa-hospital-user fa-3x mb-3"></i><p>Please select a hospital to view available doctors</p></div>';
                return;
            }
            
            document.getElementById('doctors_list').innerHTML = '<div class="col-12 text-center"><i class="fas fa-spinner fa-spin fa-2x"></i><p>Loading doctors...</p></div>';
            
            fetch('<?php echo base_url('frontend/get_doctors_by_hospital'); ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: 'hospital_id=' + hospitalId
            })
            .then(response => response.json())
            .then(data => {
                availableDoctors = data.doctors || [];
                displayDoctors(availableDoctors);
            })
            .catch(error => {
                console.error('Error:', error);
                document.getElementById('doctors_list').innerHTML = '<div class="col-12 text-center text-danger"><i class="fas fa-exclamation-triangle fa-2x mb-3"></i><p>Error loading doctors</p></div>';
            });
        }

        function filterDoctors() {
            const specialty = document.getElementById('specialty_filter').value;
            const filtered = specialty ? 
                availableDoctors.filter(doctor => doctor.speciality === specialty) : 
                availableDoctors;
            displayDoctors(filtered);
        }

        function displayDoctors(doctors) {
            if (doctors.length === 0) {
                document.getElementById('doctors_list').innerHTML = '<div class="col-12 text-center text-muted"><i class="fas fa-user-md fa-3x mb-3"></i><p>No doctors available</p></div>';
                return;
            }
            
            let html = '';
            doctors.forEach(doctor => {
                html += `
                    <div class="col-lg-6">
                        <div class="doctor-card bg-white p-4 border rounded-lg" onclick="selectDoctor(${doctor.id})">
                            <div class="d-flex align-items-center">
                                <div class="flex-shrink-0">
                                    <div class="w-16 h-16 bg-primary rounded-full d-flex align-items-center justify-content-center">
                                        <i class="fas fa-user-md text-white fa-lg"></i>
                                    </div>
                                </div>
                                <div class="flex-grow-1 ms-3">
                                    <h5 class="mb-1">${doctor.name}</h5>
                                    <p class="text-muted mb-1">${doctor.speciality || 'General'}</p>
                                    <p class="text-sm text-secondary mb-0">Fee: ${doctor.price ? '$' + doctor.price : 'Contact for fee'}</p>
                                </div>
                            </div>
                        </div>
                    </div>
                `;
            });
            document.getElementById('doctors_list').innerHTML = html;
        }

        function selectDoctor(doctorId) {
            // Remove previous selection
            document.querySelectorAll('.doctor-card').forEach(card => card.classList.remove('selected'));
            
            // Find and select the doctor
            selectedDoctor = availableDoctors.find(doctor => doctor.id == doctorId);
            
            // Highlight selected card
            event.currentTarget.classList.add('selected');
            
            // Enable continue button
            document.getElementById('continueToDateTime').disabled = false;
        }

        function loadAvailableDates() {
            if (!selectedDoctor) return;
            
            // Initialize flatpickr for date selection
            flatpickr("#appointment_date", {
                dateFormat: "Y-m-d",
                minDate: "today",
                maxDate: new Date().fp_incr(30), // 30 days from today
                disable: [
                    function(date) {
                        // Disable Sundays (0) - you can customize this
                        return date.getDay() === 0;
                    }
                ],
                onChange: function(selectedDates, dateStr) {
                    selectedDate = dateStr;
                    loadTimeSlots(dateStr);
                }
            });
            
            // Update selected doctor info
            document.getElementById('selected_doctor_info').innerHTML = `
                <h6>${selectedDoctor.name}</h6>
                <p class="text-muted mb-0">${selectedDoctor.speciality || 'General'}</p>
                <p class="text-sm">Fee: ${selectedDoctor.price ? '$' + selectedDoctor.price : 'Contact for fee'}</p>
            `;
        }

        function loadTimeSlots(date) {
            if (!selectedDoctor || !date) return;
            
            document.getElementById('time_slots_container').innerHTML = '<div class="text-center"><i class="fas fa-spinner fa-spin"></i> Loading time slots...</div>';
            
            fetch('<?php echo base_url('frontend/get_available_time_slots'); ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: `doctor_id=${selectedDoctor.id}&date=${date}`
            })
            .then(response => response.json())
            .then(data => {
                displayTimeSlots(data.slots || []);
            })
            .catch(error => {
                console.error('Error:', error);
                document.getElementById('time_slots_container').innerHTML = '<p class="text-danger">Error loading time slots</p>';
            });
        }

        function displayTimeSlots(slots) {
            if (slots.length === 0) {
                document.getElementById('time_slots_container').innerHTML = '<p class="text-muted">No available time slots for this date</p>';
                return;
            }
            
            let html = '<div class="row g-2">';
            slots.forEach(slot => {
                html += `
                    <div class="col-6 col-md-4">
                        <div class="time-slot text-center" onclick="selectTimeSlot('${slot}')">
                            ${slot}
                        </div>
                    </div>
                `;
            });
            html += '</div>';
            
            document.getElementById('time_slots_container').innerHTML = html;
        }

        function selectTimeSlot(time) {
            // Remove previous selection
            document.querySelectorAll('.time-slot').forEach(slot => slot.classList.remove('selected'));
            
            // Select current slot
            event.currentTarget.classList.add('selected');
            selectedTime = time;
            
            // Enable continue button
            document.getElementById('continueToConfirmation').disabled = false;
        }

        function updateAppointmentSummary() {
            const guestData = {
                name: document.getElementById('guest_name').value,
                phone: document.getElementById('guest_phone').value,
                email: document.getElementById('guest_email').value,
                age: document.getElementById('guest_age').value,
                gender: document.getElementById('guest_gender').value,
                address: document.getElementById('guest_address').value,
                symptoms: document.getElementById('guest_symptoms').value
            };
            
            const hospital = document.getElementById('hospital_select').options[document.getElementById('hospital_select').selectedIndex].text;
            
            document.getElementById('appointment_summary').innerHTML = `
                <div class="row g-4">
                    <div class="col-md-6">
                        <h6 class="fw-bold text-primary">Patient Information</h6>
                        <p><strong>Name:</strong> ${guestData.name}</p>
                        <p><strong>Phone:</strong> ${guestData.phone}</p>
                        <p><strong>Age:</strong> ${guestData.age} years</p>
                        <p><strong>Gender:</strong> ${guestData.gender}</p>
                        ${guestData.symptoms ? `<p><strong>Symptoms:</strong> ${guestData.symptoms}</p>` : ''}
                    </div>
                    <div class="col-md-6">
                        <h6 class="fw-bold text-primary">Appointment Details</h6>
                        <p><strong>Doctor:</strong> ${selectedDoctor.name}</p>
                        <p><strong>Specialty:</strong> ${selectedDoctor.speciality || 'General'}</p>
                        <p><strong>Hospital:</strong> ${hospital}</p>
                        <p><strong>Date:</strong> ${selectedDate}</p>
                        <p><strong>Time:</strong> ${selectedTime}</p>
                    </div>
                </div>
            `;
        }

        function submitGuestAppointment() {
            const appointmentData = {
                guest_name: document.getElementById('guest_name').value,
                guest_phone: document.getElementById('guest_phone').value,
                guest_email: document.getElementById('guest_email').value,
                guest_age: document.getElementById('guest_age').value,
                guest_gender: document.getElementById('guest_gender').value,
                guest_address: document.getElementById('guest_address').value,
                guest_emergency_contact: document.getElementById('guest_emergency_contact').value,
                symptoms: document.getElementById('guest_symptoms').value,
                doctor_id: selectedDoctor.id,
                hospital_id: document.getElementById('hospital_select').value,
                appointment_date: selectedDate,
                appointment_time: selectedTime
            };
            
            Swal.fire({
                title: 'Submitting Appointment...',
                text: 'Please wait while we process your appointment.',
                icon: 'info',
                allowOutsideClick: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });
            
            fetch('<?php echo base_url('frontend/submit_guest_appointment'); ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: new URLSearchParams(appointmentData)
            })
            .then(response => response.json())
            .then(data => {
                    
                    if (data.success) {
                    // Beautiful guest appointment success message
                    Swal.fire({
                        title: 'Guest Appointment Booked Successfully!',
                        html: `
                            <div class="text-center">
                                <i class="fas fa-user-clock text-purple-600 text-4xl mb-4"></i>
                                <p class="text-lg mb-2">Your guest appointment has been scheduled successfully!</p>
                                <div class="bg-purple-50 p-4 rounded-lg mb-4">
                                    <p class="font-semibold text-purple-800">📅 Appointment Details:</p>
                                    <p class="text-purple-700">Appointment ID: <strong>#${data.appointment_id || 'Generated'}</strong></p>
                                    <p class="text-purple-700">Status: <strong>Pending Confirmation</strong></p>
                                    <p class="text-sm text-purple-600 mt-2">📞 Hospital will contact you for confirmation</p>
                                </div>
                                <div class="bg-blue-50 p-4 rounded-lg mb-4">
                                    <p class="font-semibold text-blue-800">📋 Next Steps:</p>
                                    <p class="text-blue-700 text-sm">• Hospital will call you to confirm the appointment</p>
                                    <p class="text-blue-700 text-sm">• Please keep your phone available</p>
                                    <p class="text-blue-700 text-sm">• Bring a valid ID to your appointment</p>
                                    <p class="text-blue-700 text-sm">• Payment can be made at the hospital</p>
                                </div>
                                <div class="bg-yellow-50 p-4 rounded-lg mb-4">
                                    <p class="font-semibold text-yellow-800">ℹ️ Guest Appointment Notice</p>
                                    <p class="text-yellow-700 text-sm">As a guest, your appointment requires manual confirmation by hospital staff.</p>
                                </div>
                                <p class="text-gray-600">View your appointment details for more information!</p>
                            </div>
                        `,
                        icon: 'success',
                        showCancelButton: true,
                        confirmButtonText: '<i class="fas fa-eye mr-2"></i>View Details',
                        cancelButtonText: '<i class="fas fa-home mr-2"></i>Go to Home',
                        confirmButtonColor: '#7c3aed',
                        cancelButtonColor: '#6b7280'
                    }).then((result) => {
                        if (result.isConfirmed) {
                            window.location.href = '<?php echo base_url('frontend/guest_appointment_confirmation'); ?>/' + data.appointment_id;
                        } else {
                            window.location.href = '<?php echo base_url('frontend/public_portal'); ?>';
                        }
                    });
                    } else {
                        Swal.fire('Error', data.message || 'Failed to book appointment. Please try again.', 'error');
                    }
            })
            .catch(error => {
                console.error('Error:', error);
                Swal.fire('Error', 'Something went wrong. Please try again.', 'error');
            });
        }

        // Phone verification variables
        let phoneCheckTimeout;
        let lastCheckedPhone = '';
        
        // Handle phone input with debouncing
        function handlePhoneInput() {
            const phone = document.getElementById('guest_phone').value.trim();
            
            // Clear previous timeout
            if (phoneCheckTimeout) {
                clearTimeout(phoneCheckTimeout);
            }
            
            // Hide result if phone is too short
            if (!phone || phone.length < 10) {
                $('#phoneCheckResult').hide();
                $('#phoneDefaultMessage').show();
                return;
            }
            
            // Don't check if it's the same number we already checked
            if (phone === lastCheckedPhone) {
                return;
            }
            
            // Show checking message immediately for better UX
            $('#phoneDefaultMessage').hide();
            $('#phoneCheckResult').html('<div class="alert alert-info"><i class="fas fa-spinner fa-spin"></i> Checking registration status...</div>').show();
            
            // Set timeout for automatic checking (1 second after user stops typing)
            phoneCheckTimeout = setTimeout(function() {
                checkPhoneRegistration();
            }, 1000);
        }
        
        // Phone verification function
        function checkPhoneRegistration() {
            const phone = document.getElementById('guest_phone').value.trim();
            const $result = $('#phoneCheckResult');
            
            if (!phone || phone.length < 10) {
                $result.hide();
                return;
            }
            
            // Don't check if it's the same number we already checked
            if (phone === lastCheckedPhone) {
                return;
            }
            
            lastCheckedPhone = phone;
            
            $.ajax({
                url: '<?php echo base_url('frontend/check_existing_patient'); ?>',
                method: 'POST',
                data: { phone: phone },
                dataType: 'json',
                success: function(response) {
                    if (response.exists) {
                        if (response.is_patient) {
                            // Patient is registered
                            $result.html(`
                                <div class="alert alert-warning border-left-warning">
                                    <div class="d-flex align-items-center">
                                        <i class="fas fa-user-check fa-2x text-warning mr-3"></i>
                                        <div>
                                            <h6 class="font-weight-bold mb-1">Already Registered as Patient!</h6>
                                            <p class="mb-2">This phone number is registered to: <strong>${response.patient.name}</strong></p>
                                            <p class="mb-3">You can book appointments directly with your account.</p>
                                            <div class="d-flex flex-column flex-sm-row gap-2">
                                                <a href="<?php echo base_url('frontend/appointment_request'); ?>" 
                                                   class="btn btn-primary btn-sm flex-fill">
                                                    <i class="fas fa-calendar-check mr-1"></i>Book Regular Appointment
                                                </a>
                                                <button type="button" class="btn btn-secondary btn-sm flex-fill" onclick="continueAsGuest()">
                                                    <i class="fas fa-user-clock mr-1"></i>Continue as Guest
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            `);
                        } else {
                            // Phone belongs to other user type
                            $result.html(`
                                <div class="alert alert-danger border-left-danger">
                                    <div class="d-flex align-items-center">
                                        <i class="fas fa-exclamation-triangle fa-2x text-danger mr-3"></i>
                                        <div>
                                            <h6 class="font-weight-bold mb-1">Phone Number Already Used!</h6>
                                            <p class="mb-0">This mobile number is registered by another type of user (Doctor, Admin, etc.). 
                                            Please use a different phone number or contact support if this is your number.</p>
                                        </div>
                                    </div>
                                </div>
                            `);
                        }
                    } else {
                        // Phone not registered - good to continue
                        $result.html(`
                            <div class="alert alert-success border-left-success">
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-check-circle fa-2x text-success mr-3"></i>
                                    <div>
                                        <h6 class="font-weight-bold mb-1">Phone Number Available!</h6>
                                        <p class="mb-0">This number is not registered. You can continue with guest appointment booking.</p>
                                    </div>
                                </div>
                            </div>
                        `);
                    }
                    $result.show();
                },
                error: function() {
                    $result.html(`
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-triangle mr-2"></i>
                            Error checking phone number. Please try again.
                        </div>
                    `).show();
                }
            });
        }
        
        function continueAsGuest() {
            $('#phoneCheckResult').hide();
            $('#phoneDefaultMessage').show();
            // Allow user to continue with guest booking
        }

        // Initialize when page loads
        document.addEventListener('DOMContentLoaded', function() {
            // Set up event listeners for step 3
            const step3Observer = new MutationObserver(function(mutations) {
                mutations.forEach(function(mutation) {
                    if (mutation.target.id === 'step-3' && mutation.target.classList.contains('active')) {
                        loadAvailableDates();
                    }
                });
            });
            
            const step3Element = document.getElementById('step-3');
            if (step3Element) {
                step3Observer.observe(step3Element, { attributes: true, attributeFilter: ['class'] });
            }
        });
    </script>

    <!-- jQuery -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <!-- Include OTP Verification Modal -->
<?php $this->load->view('includes/otp_verification_modal'); ?>

<!-- Guest Appointment OTP Modal (Pure Tailwind) -->
<div id="guestOtpModal" class="fixed inset-0 z-50 hidden" aria-labelledby="guestOtpModalLabel" role="dialog" aria-modal="true">
    <!-- Background overlay -->
    <div class="fixed inset-0 bg-gray-500 bg-opacity-75 modal-overlay" onclick="closeGuestOtpModal()"></div>
    
    <!-- Modal container - perfectly centered -->
    <div class="fixed inset-0 flex items-center justify-center p-4">
        <!-- Modal panel -->
        <div class="bg-white rounded-lg shadow-xl max-w-lg w-full max-h-screen overflow-y-auto">
            <!-- Modal Header -->
            <div class="bg-gradient-to-r from-purple-600 to-green-600 px-6 py-4">
                <div class="flex items-center justify-between">
                    <h3 class="text-lg font-medium text-white" id="guestOtpModalLabel">
                        <i class="fas fa-user-clock mr-2"></i>Guest Phone Verification
                    </h3>
                    <button type="button" class="text-white hover:text-gray-200 focus:outline-none" onclick="closeGuestOtpModal()">
                        <span class="sr-only">Close</span>
                        <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                        </svg>
                    </button>
                </div>
            </div>
            
            <!-- Modal Body -->
            <div class="px-6 py-6">
                <!-- Step 1: Send OTP -->
                <div id="guest_otp_step_1" class="text-center">
                    <div class="w-20 h-20 bg-gradient-to-r from-purple-500 to-green-600 rounded-full flex items-center justify-center mx-auto mb-4">
                        <i class="fas fa-mobile-alt text-3xl text-white"></i>
                    </div>
                    <h4 class="text-lg font-semibold text-gray-800 mb-3">Verify Your Phone Number</h4>
                    <p class="text-gray-600 mb-3">We need to verify your phone number to book your guest appointment.</p>
                    <p class="text-lg font-semibold text-purple-600 mb-6" id="guest_phone_display"></p>
                    
                    <button type="button" class="bg-purple-600 hover:bg-purple-700 text-white font-medium py-3 px-8 rounded-lg transition duration-200" id="send_guest_otp_btn">
                        <i class="fas fa-paper-plane mr-2"></i>Send Verification Code
                    </button>
                </div>

                <!-- Step 2: Enter OTP -->
                <div id="guest_otp_step_2" class="text-center hidden">
                    <div class="w-20 h-20 bg-gradient-to-r from-green-500 to-purple-600 rounded-full flex items-center justify-center mx-auto mb-4">
                        <i class="fas fa-sms text-3xl text-white"></i>
                    </div>
                    <h4 class="text-lg font-semibold text-gray-800 mb-2">Verification Code Sent</h4>
                    <p class="text-gray-600 mb-1">Enter the 6-digit code sent to</p>
                    <p class="text-lg font-semibold text-purple-600 mb-6" id="guest_otp_phone_display"></p>

                    <!-- OTP Input -->
                    <form id="guestOtpVerificationForm">
                        <div class="flex justify-center gap-2 mb-4">
                            <input type="text" class="guest-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none transition duration-200" maxlength="1" id="guest_otp1" data-index="1" tabindex="1">
                            <input type="text" class="guest-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none transition duration-200" maxlength="1" id="guest_otp2" data-index="2" tabindex="2">
                            <input type="text" class="guest-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none transition duration-200" maxlength="1" id="guest_otp3" data-index="3" tabindex="3">
                            <input type="text" class="guest-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none transition duration-200" maxlength="1" id="guest_otp4" data-index="4" tabindex="4">
                            <input type="text" class="guest-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none transition duration-200" maxlength="1" id="guest_otp5" data-index="5" tabindex="5">
                            <input type="text" class="guest-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none transition duration-200" maxlength="1" id="guest_otp6" data-index="6" tabindex="6">
                        </div>
                        <p class="text-sm text-gray-500 mb-4">Enter the 6-digit verification code</p>

                        <!-- Timer and Resend -->
                        <div class="mb-4">
                            <div id="guest_otp_timer_container" class="hidden">
                                <p class="text-gray-600 mb-2">
                                    <i class="fas fa-clock mr-1"></i>
                                    Code expires in: <span id="guest_otp_timer" class="font-semibold text-yellow-600">05:00</span>
                                </p>
                            </div>
                            <div id="guest_otp_resend_container" class="hidden">
                                <p class="text-gray-600 mb-2">Didn't receive the code?</p>
                                <button type="button" class="text-purple-600 hover:text-purple-800 font-semibold focus:outline-none" id="guest_resend_otp_btn">
                                    <i class="fas fa-redo mr-1"></i>Resend Code
                                </button>
                            </div>
                        </div>

                        <!-- Message Area -->
                        <div id="guest_otp_message" class="hidden p-3 rounded-lg mb-4"></div>

                        <!-- Action Buttons -->
                        <div class="flex justify-center gap-4">
                            <button type="button" class="bg-gray-500 hover:bg-gray-600 text-white font-medium py-2 px-6 rounded-lg transition duration-200" onclick="showGuestOtpStep1()">
                                <i class="fas fa-arrow-left mr-2"></i>Back
                            </button>
                            <button type="submit" class="bg-green-600 hover:bg-green-700 text-white font-medium py-2 px-6 rounded-lg transition duration-200" id="verify_guest_otp_btn">
                                <i class="fas fa-check mr-2"></i>Verify Code
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>
    
    <script>
        // Track OTP verification status
        let otpVerifiedPhone = null;
        
        // Guest OTP Modal Variables
        let currentGuestPhone = '';
        let currentGuestPurpose = '';
        let guestOtpTimer = null;
        let guestOtpTimeLeft = 300; // 5 minutes

        // Show Guest OTP Modal (Step 1)
        function showGuestOtpModal(phone, purpose) {
            currentGuestPhone = phone;
            currentGuestPurpose = purpose;
            
            $('#guest_phone_display').text(phone);
            $('#guest_otp_phone_display').text(phone);
            
            // Show step 1, hide step 2
            showGuestOtpStep1();
            
            // Show modal with Tailwind classes
            $('#guestOtpModal').removeClass('hidden');
            $('body').addClass('overflow-hidden'); // Prevent background scrolling
        }

        // Close Guest OTP Modal
        function closeGuestOtpModal() {
            $('#guestOtpModal').addClass('hidden');
            $('body').removeClass('overflow-hidden'); // Restore background scrolling
            
            // Clear timer if running
            if (guestOtpTimer) {
                clearInterval(guestOtpTimer);
                guestOtpTimer = null;
            }
            
            // Reset form
            resetGuestOtpForm();
        }

        // Show Step 1 (Send OTP button)
        function showGuestOtpStep1() {
            $('#guest_otp_step_1').removeClass('hidden');
            $('#guest_otp_step_2').addClass('hidden');
            $('#guestOtpModalLabel').html('<i class="fas fa-user-clock mr-2"></i>Guest Phone Verification');
        }

        // Show Step 2 (OTP input fields)
        function showGuestOtpStep2() {
            $('#guest_otp_step_1').addClass('hidden');
            $('#guest_otp_step_2').removeClass('hidden');
            $('#guestOtpModalLabel').html('<i class="fas fa-sms mr-2"></i>Enter Verification Code');
            
            // Focus on first OTP input
            setTimeout(() => {
                $('#guest_otp1').focus();
            }, 100);
            
            // Start timer
            startGuestOtpTimer();
        }

        // Reset OTP form
        function resetGuestOtpForm() {
            // Clear OTP inputs
            for (let i = 1; i <= 6; i++) {
                $('#guest_otp' + i).val('');
            }
            
            // Hide message
            $('#guest_otp_message').addClass('hidden');
            
            // Reset timer containers
            $('#guest_otp_timer_container').addClass('hidden');
            $('#guest_otp_resend_container').addClass('hidden');
            
            // Reset button states
            $('#send_guest_otp_btn').prop('disabled', false).html('<i class="fas fa-paper-plane mr-2"></i>Send Verification Code');
            $('#verify_guest_otp_btn').prop('disabled', false).html('<i class="fas fa-check mr-2"></i>Verify Code');
        }

        // Send OTP (triggered by button click in step 1)
        function sendGuestOtp() {
            $('#send_guest_otp_btn').prop('disabled', true).html('<i class="fas fa-spinner fa-spin mr-2"></i>Sending...');
            
            $.ajax({
                url: '<?php echo base_url('frontend/send_otp'); ?>',
                method: 'POST',
                data: {
                    phone: currentGuestPhone,
                    purpose: 'guest_appointment'
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        showGuestOtpStep2();
                    } else {
                        showGuestOtpMessage(response.message || 'Failed to send verification code', 'danger');
                    }
                },
                error: function() {
                    showGuestOtpMessage('Network error. Please try again.', 'danger');
                },
                complete: function() {
                    $('#send_guest_otp_btn').prop('disabled', false).html('<i class="fas fa-paper-plane mr-2"></i>Send Verification Code');
                }
            });
        }

        // Timer functions
        function startGuestOtpTimer() {
            guestOtpTimeLeft = 300; // 5 minutes
            $('#guest_otp_timer_container').removeClass('hidden');
            $('#guest_otp_resend_container').addClass('hidden');
            
            guestOtpTimer = setInterval(function() {
                guestOtpTimeLeft--;
                updateGuestOtpTimer();
                
                if (guestOtpTimeLeft <= 0) {
                    clearInterval(guestOtpTimer);
                    $('#guest_otp_timer_container').addClass('hidden');
                    $('#guest_otp_resend_container').removeClass('hidden');
                }
            }, 1000);
        }

        function updateGuestOtpTimer() {
            let minutes = Math.floor(guestOtpTimeLeft / 60);
            let seconds = guestOtpTimeLeft % 60;
            $('#guest_otp_timer').text(
                (minutes < 10 ? '0' : '') + minutes + ':' + 
                (seconds < 10 ? '0' : '') + seconds
            );
        }

        // OTP input handling
        function setupGuestOtpInputs() {
            // Handle input navigation for guest OTP inputs using class
            $('.guest-otp-input').on('input', function() {
                if (this.value.length === 1) {
                    let currentIndex = parseInt($(this).data('index'));
                    if (currentIndex < 6) {
                        $('#guest_otp' + (currentIndex + 1)).focus();
                    }
                }
            });

            $('.guest-otp-input').on('keydown', function(e) {
                if (e.key === 'Backspace' && this.value.length === 0) {
                    let currentIndex = parseInt($(this).data('index'));
                    if (currentIndex > 1) {
                        $('#guest_otp' + (currentIndex - 1)).focus();
                    }
                }
            });

            $('.guest-otp-input').on('paste', function(e) {
                e.preventDefault();
                let paste = (e.originalEvent.clipboardData || window.clipboardData).getData('text');
                if (paste.length === 6 && /^\d+$/.test(paste)) {
                    $('.guest-otp-input').each(function(index) {
                        $(this).val(paste.charAt(index));
                    });
                    $('#guest_otp6').focus();
                }
            });
        }

        // Verify OTP
        function verifyGuestOtp() {
            let otp = '';
            
            // Collect OTP from inputs using class selector
            $('.guest-otp-input').each(function() {
                let value = $(this).val() || '';
                otp += value;
            });
            
            if (otp.length !== 6 || !/^\d{6}$/.test(otp)) {
                showGuestOtpMessage('Please enter the complete 6-digit code', 'danger');
                return;
            }
            
            if (!currentGuestPhone || currentGuestPhone.trim() === '') {
                showGuestOtpMessage('Phone number is missing. Please close and try again.', 'danger');
                return;
            }
            
            $('#verify_guest_otp_btn').prop('disabled', true).html('<i class="fas fa-spinner fa-spin mr-2"></i>Verifying...');
            
            $.ajax({
                url: '<?php echo base_url('frontend/verify_otp'); ?>',
                method: 'POST',
                data: {
                    phone: currentGuestPhone,
                    otp_code: otp,
                    purpose: 'guest_appointment'
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        showGuestOtpMessage('✅ Phone verified successfully!', 'success');
                        setTimeout(() => {
                            closeGuestOtpModal();
                            handleGuestOtpSuccess();
                        }, 1500);
                    } else {
                        showGuestOtpMessage(response.message || 'Invalid verification code', 'danger');
                    }
                },
                error: function() {
                    showGuestOtpMessage('Network error. Please try again.', 'danger');
                },
                complete: function() {
                    $('#verify_guest_otp_btn').prop('disabled', false).html('<i class="fas fa-check mr-2"></i>Verify Code');
                }
            });
        }

        // Handle successful OTP verification
        function handleGuestOtpSuccess() {
            otpVerifiedPhone = currentGuestPhone;
            // Show success message in phone check result
            $('#phoneCheckResult').html(`
                <div class="alert alert-success border-left-success">
                    <div class="d-flex align-items-center">
                        <i class="fas fa-check-circle fa-2x text-success mr-3"></i>
                        <div>
                            <h6 class="font-weight-bold mb-1">Phone Number Verified!</h6>
                            <p class="mb-0">Your phone number has been verified via OTP. You can now proceed with the appointment booking.</p>
                        </div>
                    </div>
                </div>
            `).show();
            $('#phoneDefaultMessage').hide();
        }

        // Show message in OTP modal
        function showGuestOtpMessage(message, type) {
            let bgColor = '';
            let textColor = '';
            let borderColor = '';
            
            if (type === 'success') {
                bgColor = 'bg-green-100';
                textColor = 'text-green-800';
                borderColor = 'border-green-400';
            } else if (type === 'danger' || type === 'error') {
                bgColor = 'bg-red-100';
                textColor = 'text-red-800';
                borderColor = 'border-red-400';
            } else if (type === 'warning') {
                bgColor = 'bg-yellow-100';
                textColor = 'text-yellow-800';
                borderColor = 'border-yellow-400';
            } else {
                bgColor = 'bg-blue-100';
                textColor = 'text-blue-800';
                borderColor = 'border-blue-400';
            }
            
            $('#guest_otp_message')
                .removeClass()
                .addClass(`${bgColor} ${textColor} ${borderColor} border p-3 rounded-lg mb-4`)
                .text(message)
                .removeClass('hidden');
                
            setTimeout(() => {
                $('#guest_otp_message').addClass('hidden');
            }, 5000);
        }

        // Initialize Guest OTP Modal
        $(document).ready(function() {
            setupGuestOtpInputs();
            
            // Send OTP button click
            $('#send_guest_otp_btn').click(function() {
                sendGuestOtp();
            });
            
            // Resend OTP button click
            $('#guest_resend_otp_btn').click(function() {
                sendGuestOtp();
            });
            
            // OTP form submission
            $('#guestOtpVerificationForm').submit(function(e) {
                e.preventDefault();
                verifyGuestOtp();
            });
            
            // Keyboard event handling
            $(document).keydown(function(e) {
                if (e.key === 'Escape' && !$('#guestOtpModal').hasClass('hidden')) {
                    closeGuestOtpModal();
                }
            });
        });
        
        // OTP verification success callback
        window.otpVerificationSuccess = function(phone, purpose) {
            if (purpose === 'guest_appointment') {
                otpVerifiedPhone = phone;
                // Show success message in phone check result
                $('#phoneCheckResult').html(`
                    <div class="alert alert-success border-left-success">
                        <div class="d-flex align-items-center">
                            <i class="fas fa-check-circle fa-2x text-success mr-3"></i>
                            <div>
                                <h6 class="font-weight-bold mb-1">Phone Number Verified!</h6>
                                <p class="mb-0">Your phone number has been verified via OTP. You can now proceed with the appointment booking.</p>
                            </div>
                        </div>
                    </div>
                `).show();
                $('#phoneDefaultMessage').hide();
            }
        };
        
        // Function to initiate OTP verification
        function initiateOTPVerification(phone) {
            // Show the new beautiful OTP modal immediately
            showGuestOtpModal(phone, 'guest_appointment');
        }
        
        // Modify the existing checkPhoneRegistration function to include OTP verification
        const originalCheckPhoneRegistration = checkPhoneRegistration;
        checkPhoneRegistration = function() {
            const phone = document.getElementById('guest_phone').value.trim();
            const $result = $('#phoneCheckResult');
            
            if (!phone || phone.length < 10) {
                $result.hide();
                $('#phoneDefaultMessage').show();
                return;
            }
            
            // Don't check if it's the same number we already checked
            if (phone === lastCheckedPhone) {
                return;
            }
            
            lastCheckedPhone = phone;
            
            $.ajax({
                url: '<?php echo base_url('frontend/check_existing_patient'); ?>',
                method: 'POST',
                data: { phone: phone },
                dataType: 'json',
                success: function(response) {
                    if (response.exists) {
                        if (response.is_patient) {
                            // Patient is registered
                            $result.html(`
                                <div class="alert alert-warning border-left-warning">
                                    <div class="d-flex align-items-center">
                                        <i class="fas fa-user-check fa-2x text-warning mr-3"></i>
                                        <div>
                                            <h6 class="font-weight-bold mb-1">Already Registered as Patient!</h6>
                                            <p class="mb-2">This phone number is registered to: <strong>${response.patient.name}</strong></p>
                                            <p class="mb-3">You can book appointments directly with your account.</p>
                                            <div class="d-flex flex-column flex-sm-row gap-2">
                                                <a href="<?php echo base_url('frontend/appointment_request'); ?>" 
                                                   class="btn btn-primary btn-sm flex-fill">
                                                    <i class="fas fa-calendar-check mr-1"></i>Book Regular Appointment
                                                </a>
                                                <button type="button" class="btn btn-secondary btn-sm flex-fill" onclick="continueAsGuest()">
                                                    <i class="fas fa-user-clock mr-1"></i>Continue as Guest
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            `);
                        } else {
                            // Phone belongs to other user type
                            $result.html(`
                                <div class="alert alert-danger border-left-danger">
                                    <div class="d-flex align-items-center">
                                        <i class="fas fa-exclamation-triangle fa-2x text-danger mr-3"></i>
                                        <div>
                                            <h6 class="font-weight-bold mb-1">Phone Number Already Used!</h6>
                                            <p class="mb-0">This mobile number is registered by another type of user (Doctor, Admin, etc.). 
                                            Please use a different phone number or contact support if this is your number.</p>
                                        </div>
                                    </div>
                                </div>
                            `);
                        }
                    } else {
                        // Phone not registered - require OTP verification for guest appointments
                        $result.html(`
                            <div class="alert alert-info border-left-info">
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-shield-alt fa-2x text-info mr-3"></i>
                                    <div>
                                        <h6 class="font-weight-bold mb-1">Verification Required</h6>
                                        <p class="mb-3">For security, we need to verify your phone number before proceeding with the guest appointment.</p>
                                        <button type="button" class="btn btn-info btn-sm" onclick="initiateOTPVerification('${phone}')">
                                            <i class="fas fa-mobile-alt mr-1"></i>Verify Phone Number
                                        </button>
                                    </div>
                                </div>
                            </div>
                        `);
                    }
                    $result.show();
                    $('#phoneDefaultMessage').hide();
                },
                error: function() {
                    $result.html(`
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-triangle mr-2"></i>
                            Error checking phone number. Please try again.
                        </div>
                    `).show();
                }
            });
        }
        
        // Modify the form submission to check for OTP verification
        const originalGoToStep = goToStep;
        goToStep = function(step) {
            if (step === 2) { // Going from step 1 (personal info) to step 2 (doctor selection)
                const phone = document.getElementById('guest_phone').value.trim();
                
                // Check if phone is filled
                if (!phone || phone.length < 10) {
                    Swal.fire('Validation Error', 'Please enter a valid phone number', 'error');
                    return;
                }
                
                // Check if OTP verification is required and completed
                if (phone !== otpVerifiedPhone) {
                    // Check if user exists (if they exist, OTP not needed)
                    $.ajax({
                        url: '<?php echo base_url('frontend/check_existing_patient'); ?>',
                        method: 'POST',
                        data: { phone: phone },
                        dataType: 'json',
                        success: function(response) {
                            if (response.exists && response.is_patient) {
                                // User exists, can proceed without OTP
                                originalGoToStep(step);
                            } else if (!response.exists) {
                                // New user, must verify OTP
                                Swal.fire({
                                    title: 'Phone Verification Required',
                                    text: 'Please verify your phone number first before proceeding.',
                                    icon: 'warning',
                                    confirmButtonText: 'Verify Now',
                                    showCancelButton: true,
                                    cancelButtonText: 'Cancel'
                                }).then((result) => {
                                    if (result.isConfirmed) {
                                        initiateOTPVerification(phone);
                                    }
                                });
                            } else {
                                // Registered user type (doctor/admin), can proceed
                                originalGoToStep(step);
                            }
                        },
                        error: function() {
                            Swal.fire('Error', 'Unable to verify phone number status. Please try again.', 'error');
                        }
                    });
                } else {
                    // OTP verified, proceed normally
                    originalGoToStep(step);
                }
            } else {
                // Other steps, proceed normally
                originalGoToStep(step);
            }
        }
    </script>
</body>
</html>