<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * OTP Controller for Frontend
 * Handles OTP generation, sending, and verification for public forms
 */
class Otp extends MX_Controller {
    
    public function __construct() {
        parent::__construct();
        $this->load->library('otp_lib');
        $this->load->model('settings_model');
    }
    
    /**
     * Send OTP to phone number
     * POST: phone, purpose, hospital_id (optional)
     */
    public function send_otp() {
        header('Content-Type: application/json');
        
        try {
            $phone = $this->input->post('phone');
            $purpose = $this->input->post('purpose');
            $hospital_id = $this->input->post('hospital_id');
            
            // Validation
            if (empty($phone)) {
                echo json_encode(array('success' => false, 'message' => 'Phone number is required'));
                return;
            }
            
            if (empty($purpose)) {
                echo json_encode(array('success' => false, 'message' => 'Purpose is required'));
                return;
            }
            
            // Validate purpose
            $valid_purposes = ['patient_registration', 'doctor_registration', 'guest_appointment', 'login_verification', 'phone_change'];
            if (!in_array($purpose, $valid_purposes)) {
                echo json_encode(array('success' => false, 'message' => 'Invalid purpose'));
                return;
            }
            
            // Get hospital_id from settings if not provided
            if (empty($hospital_id)) {
                $hospital_id = $this->settings_model->getSettings()->hospital_id ?? 1;
            }
            
            // Generate and send OTP
            $result = $this->otp_lib->generateAndSendOTP($phone, $purpose, $hospital_id);
            
            echo json_encode($result);
            
        } catch (Exception $e) {
            log_message('error', 'OTP Send Error: ' . $e->getMessage());
            echo json_encode(array('success' => false, 'message' => 'System error occurred'));
        }
    }
    
    /**
     * Verify OTP code
     * POST: phone, otp_code, purpose
     */
    public function verify_otp() {
        header('Content-Type: application/json');
        
        try {
            $phone = $this->input->post('phone');
            $otp_code = $this->input->post('otp_code');
            $purpose = $this->input->post('purpose');
            
            // Validation
            if (empty($phone)) {
                echo json_encode(array('success' => false, 'message' => 'Phone number is required'));
                return;
            }
            
            if (empty($otp_code)) {
                echo json_encode(array('success' => false, 'message' => 'OTP code is required'));
                return;
            }
            
            if (empty($purpose)) {
                echo json_encode(array('success' => false, 'message' => 'Purpose is required'));
                return;
            }
            
            // Verify OTP
            $result = $this->otp_lib->verifyOTP($phone, $otp_code, $purpose);
            
            echo json_encode($result);
            
        } catch (Exception $e) {
            log_message('error', 'OTP Verify Error: ' . $e->getMessage());
            echo json_encode(array('success' => false, 'message' => 'System error occurred'));
        }
    }
    
    /**
     * Check if phone has verified OTP for specific purpose
     * POST: phone, purpose, within_minutes (optional, default 30)
     */
    public function check_verified() {
        header('Content-Type: application/json');
        
        try {
            $phone = $this->input->post('phone');
            $purpose = $this->input->post('purpose');
            $within_minutes = $this->input->post('within_minutes') ?: 30;
            
            // Validation
            if (empty($phone)) {
                echo json_encode(array('success' => false, 'message' => 'Phone number is required'));
                return;
            }
            
            if (empty($purpose)) {
                echo json_encode(array('success' => false, 'message' => 'Purpose is required'));
                return;
            }
            
            // Check verification status
            $is_verified = $this->otp_lib->hasVerifiedOTP($phone, $purpose, $within_minutes);
            
            echo json_encode(array(
                'success' => true,
                'verified' => $is_verified,
                'message' => $is_verified ? 'Phone number is verified' : 'Phone number not verified'
            ));
            
        } catch (Exception $e) {
            log_message('error', 'OTP Check Error: ' . $e->getMessage());
            echo json_encode(array('success' => false, 'message' => 'System error occurred'));
        }
    }
    
    /**
     * Resend OTP (same as send_otp but with explicit resend context)
     */
    public function resend_otp() {
        $this->send_otp();
    }
}