<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Settings extends MX_Controller
{

    function __construct()
    {
        parent::__construct();
        $this->load->model('doctor_model');
        $this->load->model('ion_auth_model');
        if (!$this->ion_auth->in_group(array('Doctor'))) {
            redirect('home/permission');
        }
    }

    public function index()
    {
        $doctor_ion_id = $this->ion_auth->get_user_id();
        $doctor = $this->doctor_model->getDoctorByIonUserId($doctor_ion_id);
        
        if (!$doctor) {
            redirect('home/permission');
        }

        $data['doctor'] = $doctor;
        $data['settings'] = $this->settings_model->getSettings();
        
        $this->load->view('home/dashboard', $data);
        $this->load->view('settings', $data);
        $this->load->view('home/footer');
    }

    public function updateScheduleType()
    {
        $doctor_ion_id = $this->ion_auth->get_user_id();
        $doctor = $this->doctor_model->getDoctorByIonUserId($doctor_ion_id);
        
        if (!$doctor) {
            redirect('home/permission');
        }

        $schedule_type = $this->input->post('schedule_type');
        
        $this->load->library('form_validation');
        $this->form_validation->set_rules('schedule_type', 'Schedule Type', 'trim|required|in_list[weekday,date]|xss_clean');

        if ($this->form_validation->run() == FALSE) {
            show_swal('error', lang('invalid_schedule_type'));
        } else {
            $update_data = array(
                'schedule_type' => $schedule_type
            );
            
            $this->doctor_model->updateDoctor($doctor->id, $update_data);
            
            show_swal('success', lang('schedule_type_updated_successfully'));
        }
        
        redirect('doctor/settings');
    }

    public function updateOnlinePayment()
    {
        $doctor_ion_id = $this->ion_auth->get_user_id();
        $doctor = $this->doctor_model->getDoctorByIonUserId($doctor_ion_id);
        
        if (!$doctor) {
            show_swal('error', 'Doctor not found');
            redirect('home/permission');
        }

        $online_payment_enabled = $this->input->post('online_payment_enabled');
        
        // Debug logging
        log_message('debug', 'Received online_payment_enabled: ' . $online_payment_enabled);
        log_message('debug', 'Doctor ID: ' . $doctor->id);
        
        $this->load->library('form_validation');
        $this->form_validation->set_rules('online_payment_enabled', 'Online Payment', 'trim|required|in_list[required,optional,disabled]|xss_clean');

        if ($this->form_validation->run() == FALSE) {
            show_swal('error', 'Invalid online payment setting: ' . validation_errors());
        } else {
            $update_data = array(
                'online_payment_enabled' => $online_payment_enabled
            );
            
            // Debug the update
            log_message('debug', 'Updating doctor with data: ' . json_encode($update_data));
            
            $result = $this->doctor_model->updateDoctor($doctor->id, $update_data);
            
            // Check if update was successful
            $updated_doctor = $this->doctor_model->getDoctorById($doctor->id);
            log_message('debug', 'Updated doctor payment setting: ' . $updated_doctor->online_payment_enabled);
            
            $status_messages = array(
                'required' => 'Payment is now required for all appointments',
                'optional' => 'Payment is now optional (patient choice)',
                'disabled' => 'Appointments can be booked without payment'
            );
            show_swal('success', $status_messages[$online_payment_enabled]);
        }
        
        redirect('doctor/settings');
    }

    public function changePassword()
    {
        $doctor_ion_id = $this->ion_auth->get_user_id();
        $doctor = $this->doctor_model->getDoctorByIonUserId($doctor_ion_id);
        
        if (!$doctor) {
            redirect('home/permission');
        }

        $current_password = $this->input->post('current_password');
        $new_password = $this->input->post('new_password');
        $confirm_password = $this->input->post('confirm_password');
        
        $this->load->library('form_validation');
        $this->form_validation->set_rules('current_password', 'Current Password', 'trim|required|xss_clean');
        $this->form_validation->set_rules('new_password', 'New Password', 'trim|required|min_length[5]|xss_clean');
        $this->form_validation->set_rules('confirm_password', 'Confirm Password', 'trim|required|matches[new_password]|xss_clean');

        if ($this->form_validation->run() == FALSE) {
            show_swal('error', validation_errors());
        } else {
            // Verify current password
            $ion_user_id = $doctor->ion_user_id;
            $current_user = $this->db->get_where('users', array('id' => $ion_user_id))->row();
            
            if (!$current_user || !$this->ion_auth_model->verify_password($current_password, $current_user->password)) {
                show_swal('error', 'Current password is incorrect');
            } else {
                // Update password
                $hashed_password = $this->ion_auth_model->hash_password($new_password);
                $update_data = array('password' => $hashed_password);
                
                $this->db->where('id', $ion_user_id);
                $this->db->update('users', $update_data);
                
                show_swal('success', 'Password changed successfully');
            }
        }
        
        redirect('doctor/settings');
    }

    public function updateCancellationPermission()
    {
        $doctor_ion_id = $this->ion_auth->get_user_id();
        $doctor = $this->doctor_model->getDoctorByIonUserId($doctor_ion_id);
        
        if (!$doctor) {
            show_swal('error', 'Doctor not found');
            redirect('home/permission');
        }

        $cancellation_permission = $this->input->post('cancellation_permission');
        $cancellation_type = $this->input->post('cancellation_type');
        
        // Debug logging
        log_message('debug', 'Received cancellation_permission: ' . $cancellation_permission);
        log_message('debug', 'Received cancellation_type: ' . $cancellation_type);
        log_message('debug', 'Doctor ID: ' . $doctor->id);
        
        $this->load->library('form_validation');
        $this->form_validation->set_rules('cancellation_permission', 'Cancellation Permission', 'trim|required|in_list[allow,disallow]|xss_clean');
        
        // Only validate cancellation_type if cancellation is allowed
        if ($cancellation_permission === 'allow') {
            $this->form_validation->set_rules('cancellation_type', 'Cancellation Type', 'trim|required|in_list[paid_only,non_paid_only,both]|xss_clean');
        } else {
            // Set default value when disallowed
            $cancellation_type = 'both';
        }

        if ($this->form_validation->run() == FALSE) {
            show_swal('error', 'Invalid cancellation permission setting: ' . validation_errors());
        } else {
            $update_data = array(
                'cancellation_permission' => $cancellation_permission,
                'cancellation_type' => $cancellation_type
            );
            
            // Debug the update
            log_message('debug', 'Updating doctor with cancellation data: ' . json_encode($update_data));
            
            $result = $this->doctor_model->updateDoctor($doctor->id, $update_data);
            
            // Check if update was successful
            $updated_doctor = $this->doctor_model->getDoctorById($doctor->id);
            log_message('debug', 'Updated doctor cancellation setting: ' . $updated_doctor->cancellation_permission);
            
            $status_messages = array(
                'allow' => 'Patients can now cancel their appointments',
                'disallow' => 'Patients cannot cancel appointments once booked'
            );
            show_swal('success', $status_messages[$cancellation_permission]);
        }
        
        redirect('doctor/settings');
    }
    
    public function updateSmsSettings()
    {
        $doctor_ion_id = $this->ion_auth->get_user_id();
        $doctor = $this->doctor_model->getDoctorByIonUserId($doctor_ion_id);
        
        if (!$doctor) {
            show_swal('error', 'Doctor not found');
            redirect('home/permission');
        }

        // Get checkbox values (if unchecked, they won't be in POST)
        $sms_advanced_paid_bookings = $this->input->post('sms_advanced_paid_bookings') ? 'yes' : 'no';
        $sms_online_no_payment = $this->input->post('sms_online_no_payment') ? 'yes' : 'no';
        $sms_ivr_bookings = $this->input->post('sms_ivr_bookings') ? 'yes' : 'no';
        $sms_session_start_notification_patient = $this->input->post('sms_session_start_notification_patient') ? 'yes' : 'no';
        
        $update_data = array(
            'sms_advanced_paid_bookings' => $sms_advanced_paid_bookings,
            'sms_online_no_payment' => $sms_online_no_payment,
            'sms_ivr_bookings' => $sms_ivr_bookings,
            'sms_session_start_notification_patient' => $sms_session_start_notification_patient
        );
        
        $result = $this->doctor_model->updateDoctor($doctor->id, $update_data);
        
        if ($result) {
            show_swal('success', 'SMS notification settings updated successfully');
        } else {
            show_swal('error', 'Failed to update SMS settings');
        }
        
        redirect('doctor/settings');
    }
} 