<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * OTP Library for Hospital Management System
 * Handles OTP generation, sending, and verification
 * Integrates with existing SMS system
 */
class Otp_lib {
    
    private $CI;
    
    public function __construct() {
        $this->CI =& get_instance();
        $this->CI->load->database();
        $this->CI->load->model('sms/sms_model');
        $this->CI->load->model('settings_model');
    }
    
    /**
     * Generate and send OTP to phone number
     * @param string $phone Phone number
     * @param string $purpose Purpose of OTP (patient_registration, doctor_registration, guest_appointment, appointment_booking)
     * @param int $hospital_id Hospital ID
     * @return array Result with success status and message
     */
    public function generateAndSendOTP($phone, $purpose, $hospital_id = null) {
        try {
            // Log the original phone number for debugging
            log_message('debug', 'OTP: Original phone number: ' . $phone);
            
            // Clean phone number
            $cleaned_phone = $this->cleanPhoneNumber($phone);
            log_message('debug', 'OTP: Cleaned phone number: ' . $cleaned_phone);
            
            // Validate input
            if (empty($cleaned_phone) || !$this->isValidPhoneNumber($cleaned_phone)) {
                log_message('debug', 'OTP: Validation failed for phone: ' . $cleaned_phone);
                return array('success' => false, 'message' => 'Invalid phone number. Please enter a valid Bangladeshi (01XXXXXXXXX) or Sri Lankan (0XXXXXXXXX) mobile number.');
            }
            
            $phone = $cleaned_phone;
            
            if (!in_array($purpose, ['patient_registration', 'doctor_registration', 'guest_appointment', 'appointment_booking', 'login_verification', 'phone_change'])) {
                return array('success' => false, 'message' => 'Invalid OTP purpose');
            }
            
            // Simple cooldown check - prevent spam (60 seconds)
            if ($this->isInCooldownPeriod($phone, $purpose, 60)) {
                return array('success' => false, 'message' => 'Please wait 60 seconds before requesting another OTP');
            }
            
            // Generate simple 6-digit OTP
            $otp_code = $this->generateOTPCode(6);
            
            // Expires in 5 minutes
            $expires_at = date('Y-m-d H:i:s', strtotime('+5 minutes'));
            
            // Store OTP in database
            $otp_data = array(
                'phone' => $phone,
                'otp_code' => $otp_code,
                'purpose' => $purpose,
                'hospital_id' => $hospital_id,
                'is_verified' => 0,
                'created_at' => date('Y-m-d H:i:s'),
                'expires_at' => $expires_at,
                'attempts' => 0,
                'max_attempts' => $otp_settings->max_attempts,
                'ip_address' => $this->CI->input->ip_address(),
                'user_agent' => $this->CI->input->user_agent()
            );
            
            $this->CI->db->insert('otp_verification', $otp_data);
            $otp_id = $this->CI->db->insert_id();
            
            if (!$otp_id) {
                return array('success' => false, 'message' => 'Failed to store OTP');
            }
            
            // Send OTP via SMS with simple message
            $message = "Your verification code is: $otp_code. Valid for 5 minutes. Do not share this code.";
            $sms_result = $this->sendSimpleOTPSMS($phone, $message, $hospital_id);
            
            if ($sms_result['success']) {
                return array(
                    'success' => true, 
                    'message' => 'OTP sent successfully',
                    'otp_id' => $otp_id,
                    'expires_in' => '5 minutes'
                );
            } else {
                // Delete OTP if SMS failed
                $this->CI->db->delete('otp_verification', array('id' => $otp_id));
                return array('success' => false, 'message' => 'Failed to send OTP: ' . $sms_result['message']);
            }
            
        } catch (Exception $e) {
            log_message('error', 'OTP Generation Error: ' . $e->getMessage());
            return array('success' => false, 'message' => 'System error occurred');
        }
    }
    
    /**
     * Verify OTP code
     * @param string $phone Phone number
     * @param string $otp_code OTP code entered by user
     * @param string $purpose Purpose of OTP
     * @return array Result with success status and message
     */
    public function verifyOTP($phone, $otp_code, $purpose) {
        try {
            $phone = $this->cleanPhoneNumber($phone);
            
            // Find valid OTP
            $this->CI->db->where('phone', $phone);
            $this->CI->db->where('otp_code', $otp_code);
            $this->CI->db->where('purpose', $purpose);
            $this->CI->db->where('is_verified', 0);
            $this->CI->db->where('expires_at >', date('Y-m-d H:i:s'));
            $otp_record = $this->CI->db->get('otp_verification')->row();
            
            if (!$otp_record) {
                // Check if OTP exists but expired or wrong
                $this->CI->db->where('phone', $phone);
                $this->CI->db->where('purpose', $purpose);
                $this->CI->db->where('is_verified', 0);
                $this->CI->db->order_by('created_at', 'DESC');
                $this->CI->db->limit(1);
                $recent_otp = $this->CI->db->get('otp_verification')->row();
                
                if ($recent_otp) {
                    if ($recent_otp->expires_at < date('Y-m-d H:i:s')) {
                        return array('success' => false, 'message' => 'OTP has expired. Please request a new one.');
                    } else {
                        // Increment attempts
                        $this->CI->db->set('attempts', 'attempts + 1', FALSE);
                        $this->CI->db->where('id', $recent_otp->id);
                        $this->CI->db->update('otp_verification');
                        
                        if ($recent_otp->attempts >= $recent_otp->max_attempts) {
                            return array('success' => false, 'message' => 'Maximum attempts exceeded. Please request a new OTP.');
                        } else {
                            $remaining = $recent_otp->max_attempts - $recent_otp->attempts - 1;
                            return array('success' => false, 'message' => 'Invalid OTP code. ' . $remaining . ' attempts remaining.');
                        }
                    }
                }
                
                return array('success' => false, 'message' => 'Invalid OTP code');
            }
            
            // Check attempts
            if ($otp_record->attempts >= $otp_record->max_attempts) {
                return array('success' => false, 'message' => 'Maximum attempts exceeded. Please request a new OTP.');
            }
            
            // Mark as verified
            $update_data = array(
                'is_verified' => 1,
                'verified_at' => date('Y-m-d H:i:s')
            );
            
            $this->CI->db->where('id', $otp_record->id);
            $this->CI->db->update('otp_verification', $update_data);
            
            // Invalidate other OTPs for same phone and purpose
            $this->CI->db->where('phone', $phone);
            $this->CI->db->where('purpose', $purpose);
            $this->CI->db->where('id !=', $otp_record->id);
            $this->CI->db->where('is_verified', 0);
            $this->CI->db->update('otp_verification', array('is_verified' => -1)); // Mark as cancelled
            
            return array('success' => true, 'message' => 'OTP verified successfully');
            
        } catch (Exception $e) {
            log_message('error', 'OTP Verification Error: ' . $e->getMessage());
            return array('success' => false, 'message' => 'System error occurred');
        }
    }
    
    /**
     * Check if phone number has verified OTP for specific purpose
     * @param string $phone Phone number
     * @param string $purpose Purpose of OTP
     * @param int $within_minutes Check within X minutes (default 30)
     * @return boolean
     */
    public function hasVerifiedOTP($phone, $purpose, $within_minutes = 30) {
        $phone = $this->cleanPhoneNumber($phone);
        $since_time = date('Y-m-d H:i:s', strtotime('-' . $within_minutes . ' minutes'));
        
        $this->CI->db->where('phone', $phone);
        $this->CI->db->where('purpose', $purpose);
        $this->CI->db->where('is_verified', 1);
        $this->CI->db->where('verified_at >=', $since_time);
        
        return $this->CI->db->count_all_results('otp_verification') > 0;
    }
    
    /**
     * Generate random OTP code
     * @param int $length Length of OTP
     * @return string
     */
    private function generateOTPCode($length = 6) {
        $min = pow(10, $length - 1);
        $max = pow(10, $length) - 1;
        return str_pad(mt_rand($min, $max), $length, '0', STR_PAD_LEFT);
    }
    
    /**
     * Send OTP via SMS using existing SMS system
     * @param string $phone Phone number
     * @param string $otp_code OTP code
     * @param string $purpose Purpose
     * @param object $otp_settings OTP settings
     * @param int $hospital_id Hospital ID
     * @return array
     */
    private function sendOTPSMS($phone, $otp_code, $purpose, $otp_settings, $hospital_id) {
        try {
            // Get SMS gateway settings
            $sms_gateway = $this->CI->settings_model->getSettings()->sms_gateway;
            
            if (empty($sms_gateway)) {
                return array('success' => false, 'message' => 'No SMS gateway configured');
            }
            
            $sms_settings = $this->CI->sms_model->getSmsSettingsByGatewayName($sms_gateway);
            
            if (!$sms_settings) {
                return array('success' => false, 'message' => 'SMS gateway settings not found');
            }
            
            // Prepare message
            $message = str_replace(
                array('{otp_code}', '{expiry_minutes}'),
                array($otp_code, $otp_settings->otp_expiry_minutes),
                $otp_settings->message_template
            );
            
            // Send SMS based on gateway type
            $result = $this->sendSMSByGateway($sms_settings, $phone, $message);
            
            // Log SMS attempt
            $sms_log_data = array(
                'phone' => $phone,
                'message' => $message,
                'gateway' => $sms_gateway,
                'purpose' => 'OTP_' . strtoupper($purpose),
                'sent_at' => date('Y-m-d H:i:s'),
                'hospital_id' => $hospital_id,
                'status' => $result['success'] ? 'sent' : 'failed'
            );
            
            $this->CI->db->insert('sms', $sms_log_data);
            
            return $result;
            
        } catch (Exception $e) {
            log_message('error', 'SMS Send Error: ' . $e->getMessage());
            return array('success' => false, 'message' => 'Failed to send SMS');
        }
    }
    
    /**
     * Send SMS using specific gateway
     * @param object $sms_settings SMS gateway settings
     * @param string $phone Phone number
     * @param string $message Message content
     * @return array
     */
    private function sendSMSByGateway($sms_settings, $phone, $message) {
        try {
            if ($sms_settings->name == 'Clickatell') {
                $username = $sms_settings->username;
                $password = $sms_settings->password;
                $api_id = $sms_settings->api_id;
                
                $response = file_get_contents("https://api.clickatell.com/http/sendmsg?user=" . urlencode($username) . "&password=" . urlencode($password) . "&api_id=" . urlencode($api_id) . "&to=" . urlencode($phone) . "&text=" . urlencode($message));
                
                return array('success' => true, 'message' => 'SMS sent via Clickatell');
                
            } elseif ($sms_settings->name == 'MSG91') {
                $authkey = $sms_settings->authkey;
                $sender = $sms_settings->sender;
                
                $response = file_get_contents('http://world.msg91.com/api/v2/sendsms?authkey=' . urlencode($authkey) . '&mobiles=' . urlencode($phone) . '&message=' . urlencode($message) . '&sender=' . urlencode($sender) . '&route=4&country=0');
                
                return array('success' => true, 'message' => 'SMS sent via MSG91');
                
            } elseif ($sms_settings->name == 'Twilio') {
                require_once APPPATH . 'third_party/twilio/src/Twilio/autoload.php';
                
                $sid = $sms_settings->sid;
                $token = $sms_settings->token;
                $sendername = $sms_settings->sendernumber;
                
                if (!empty($sid) && !empty($token) && !empty($sendername)) {
                    $client = new Twilio\Rest\Client($sid, $token);
                    $client->messages->create(
                        $phone,
                        array(
                            'from' => $sendername,
                            'body' => $message
                        )
                    );
                    return array('success' => true, 'message' => 'SMS sent via Twilio');
                } else {
                    return array('success' => false, 'message' => 'Twilio configuration incomplete');
                }
                
            } elseif ($sms_settings->name == 'Bdbulksms') {
                $token = $sms_settings->token;
                $url = "https://api.bdbulksms.net/api.php?json";
                $data = array(
                    'to' => $phone,
                    'message' => $message,
                    'token' => $token
                );
                
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $url);
                curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
                curl_setopt($ch, CURLOPT_ENCODING, '');
                curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                
                $response = curl_exec($ch);
                curl_close($ch);
                
                return array('success' => true, 'message' => 'SMS sent via Bdbulksms');
                
            } elseif ($sms_settings->name == '80Kobo') {
                $email = $sms_settings->email;
                $password = $sms_settings->password;
                $sender_name = $sms_settings->sender_name;
                
                $data = array(
                    "email" => $email,
                    "password" => $password,
                    "message" => $message,
                    "sender_name" => $sender_name,
                    "recipients" => $phone
                );
                
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, 'https://api.80kobosms.com/v2/app/sms');
                curl_setopt($ch, CURLOPT_POST, 1);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
                curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: application/json'));
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                
                $response = curl_exec($ch);
                curl_close($ch);
                
                return array('success' => true, 'message' => 'SMS sent via 80Kobo');
                
            } elseif ($sms_settings->name == 'Hutch BSMS') {
               $username = $sms_settings->hutch_username; // your Hutch username
                $password = $sms_settings->hutch_password; // your Hutch password
                $mask     = $sms_settings->mask; // pre-approved sender ID
                $number   = $phone;
                $text  = $message;
                $campaign = "Hospital Campaign";
                // 1. Login to get access token
                $loginUrl = "https://bsms.hutch.lk/api/login";
                $loginData = [
                    "username" => $username,
                    "password" => $password
                ];

                $ch = curl_init($loginUrl);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    "Content-Type: application/json",
                    "Accept: */*",
                    "X-API-VERSION: v1"
                ]);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($loginData));
                $loginResponse = curl_exec($ch);
                curl_close($ch);

                $loginResult = json_decode($loginResponse, true);

                if (empty($loginResult['accessToken'])) {
                    echo "Login failed: " . $loginResponse;
                    return;
                }

                $accessToken = $loginResult['accessToken'];

                // 2. Send SMS using access token
                $smsUrl = "https://bsms.hutch.lk/api/sendsms";
                $smsData = [
                    "campaignName" => $campaign,
                    "mask" => $mask,
                    "numbers" => $number,
                    "content" => $text,
                    "deliveryReportRequest" => true
                ];

                $ch = curl_init($smsUrl);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    "Content-Type: application/json",
                    "Accept: */*",
                    "X-API-VERSION: v1",
                    "Authorization: Bearer " . $accessToken
                ]);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($smsData));
                $smsResponse = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);
                return array('success' => true, 'message' => 'SMS sent successfully');
                
            } elseif ($sms_settings->name == 'TextIt') {
                $username = $sms_settings->username;
                $password = $sms_settings->password;
                $sender_name = $sms_settings->sender_name;
                
                $data = array(
                    'username' => $username,
                    'password' => $password,
                    'src' => $sender_name,
                    'dst' => $phone,
                    'msg' => $message,
                    'dr' => 1
                );
                
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, 'https://www.textit.biz/sendmsg');
                curl_setopt($ch, CURLOPT_POST, 1);
                curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                
                $response = curl_exec($ch);
                curl_close($ch);
                
                return array('success' => true, 'message' => 'SMS sent via TextIt');
                
            } else {
                return array('success' => false, 'message' => 'Unsupported SMS gateway: ' . $sms_settings->name);
            }
            
        } catch (Exception $e) {
            log_message('error', 'SMS Gateway Error: ' . $e->getMessage());
            return array('success' => false, 'message' => 'Gateway communication error');
        }
    }
    
    /**
     * Get OTP settings for hospital
     * @param int $hospital_id Hospital ID
     * @return object
     */
    public function getOTPSettings($hospital_id) {
        if ($hospital_id) {
            $this->CI->db->where('hospital_id', $hospital_id);
            $settings = $this->CI->db->get('otp_settings')->row();
            
            if ($settings) {
                return $settings;
            }
        }
        
        // Return default settings if not found
        return (object) array(
            'otp_enabled' => 1,
            'otp_length' => 6,
            'otp_expiry_minutes' => 5,
            'max_attempts' => 3,
            'resend_cooldown_seconds' => 60,
            'message_template' => 'Your OTP verification code is: {otp_code}. Valid for {expiry_minutes} minutes. Do not share this code with anyone.'
        );
    }
    
    /**
     * Check if phone is in cooldown period
     * @param string $phone Phone number
     * @param string $purpose Purpose
     * @param int $cooldown_seconds Cooldown period in seconds
     * @return boolean
     */
    private function isInCooldownPeriod($phone, $purpose, $cooldown_seconds) {
        $cooldown_time = date('Y-m-d H:i:s', strtotime('-' . $cooldown_seconds . ' seconds'));
        
        $this->CI->db->where('phone', $phone);
        $this->CI->db->where('purpose', $purpose);
        $this->CI->db->where('created_at >', $cooldown_time);
        
        return $this->CI->db->count_all_results('otp_verification') > 0;
    }
    
    /**
     * Clean and format phone number
     * @param string $phone Phone number
     * @return string
     */
    private function cleanPhoneNumber($phone) {
        // Remove all non-numeric characters except +
        $phone = preg_replace('/[^\d+]/', '', $phone);
        
        // For debugging, just return the cleaned number as-is for now
        // We'll see what formats are actually being submitted
        return $phone;
    }
    
    /**
     * Validate phone number format
     * @param string $phone Phone number
     * @return boolean
     */
    private function isValidPhoneNumber($phone) {
        // Temporary permissive validation - accept any number with 8+ digits
        // This will help us debug what's happening
        return preg_match('/^\+?[0-9]{8,}$/', $phone);
    }
    
    /**
     * Clean up expired OTPs
     * @return int Number of deleted records
     */
    public function cleanupExpiredOTPs() {
        $this->CI->db->where('expires_at <', date('Y-m-d H:i:s'));
        $this->CI->db->where('is_verified', 0);
        $this->CI->db->delete('otp_verification');
        
        return $this->CI->db->affected_rows();
    }
}