<?php
$settings = $this->frontend_model->getSettings();
$title = explode(' ', $settings->title);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0" />
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <title>Book Appointment - <?php echo $settings->title; ?></title>
    
    <!-- SEO Meta Tags -->
    <meta name="description" content="Book your medical appointment online. Choose hospital, doctor specialty, doctor, date and time slot. Fast, secure, and convenient healthcare booking." />
    <meta name="keywords" content="book appointment, medical appointment, doctor appointment, hospital booking, healthcare appointment, online doctor booking, medical scheduling" />
    <meta name="author" content="<?php echo $settings->title; ?>">
    <meta name="robots" content="index, follow">
    
    <!-- Open Graph Meta Tags -->
    <meta property="og:title" content="Book Medical Appointment - <?php echo $settings->title; ?>" />
    <meta property="og:description" content="Book your medical appointment online. Fast, secure, and convenient healthcare booking." />
    <meta property="og:type" content="website">
    <meta property="og:url" content="<?php echo base_url(); ?>frontend/book_appointment">
    <meta property="og:site_name" content="<?php echo $settings->title; ?>">
    
    <!-- Twitter Meta Tags -->
    <meta name="twitter:card" content="summary">
    <meta name="twitter:title" content="Book Medical Appointment - <?php echo $settings->title; ?>">
    <meta name="twitter:description" content="Book your medical appointment online. Fast, secure, and convenient healthcare booking.">
    
    <!-- Stylesheets -->
    <link rel="stylesheet" href="new-fnt/index.css" />
    <link rel="preconnect" href="https://fonts.googleapis.com" />
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin />
    <link href="https://fonts.googleapis.com/css2?family=Manrope:wght@400;500;600;700;800&family=Source+Serif+4:wght@400;500;600;700&display=swap" rel="stylesheet">
    
    <!-- Preload critical resources -->
    <link rel="preload" href="https://code.jquery.com/jquery-3.6.0.min.js" as="script">
    <link rel="preload" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" as="style">
    
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: {
                            50: '#f0f9ff',
                            100: '#e0f2fe',
                            200: '#bae6fd',
                            300: '#7dd3fc',
                            400: '#38bdf8',
                            500: '#0ea5e9',
                            600: '#0284c7',
                            700: '#0369a1',
                            800: '#075985',
                            900: '#0c4a6e',
                        },
                        secondary: {
                            50: '#f8fafc',
                            900: '#0f172a',
                        }
                    },
                    boxShadow: {
                        'custom': '0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06)',
                        'custom-lg': '0 10px 25px -5px rgba(0, 0, 0, 0.1), 0 8px 10px -6px rgba(0, 0, 0, 0.1)',
                    }
                }
            }
        }
    </script>
    <style>
        :root {
            --ink: #0b1b2b;
            --slate: #3a4756;
            --teal: #0f766e;
            --blue: #1d4ed8;
            --mist: #e9f0f5;
            --cloud: #f7fafc;
        }

        body {
            font-family: 'Manrope', sans-serif;
            color: var(--ink);
            background: radial-gradient(circle at 15% 10%, rgba(14, 165, 168, 0.12) 0%, rgba(255,255,255,0) 40%),
                        radial-gradient(circle at 85% 0%, rgba(29, 78, 216, 0.12) 0%, rgba(255,255,255,0) 45%),
                        linear-gradient(180deg, #f8fbff 0%, #f2f6fb 100%);
        }

        .serif-head {
            font-family: 'Source Serif 4', serif;
        }
        
        .gradient-text {
            background: linear-gradient(135deg, #0f766e 0%, #1d4ed8 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        
        .form-step {
            transition: all 0.3s ease;
        }
        
        .step-indicator {
            transition: all 0.3s ease;
        }
        
        .step-indicator.active {
            background: linear-gradient(135deg, #0f766e, #1d4ed8);
            color: white;
        }
        
        .step-indicator.completed {
            background: #10b981;
            color: white;
        }
        
        .select2-container {
            width: 100% !important;
        }
        
        .datepicker {
            width: 100%;
        }
        
        .time-slot {
            transition: all 0.2s ease;
        }
        
        .time-slot:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(14, 165, 233, 0.15);
        }
        
        .time-slot.selected {
            background: linear-gradient(135deg, #0f766e, #1d4ed8);
            color: white;
            border-color: transparent;
        }
        
        .loading-spinner {
            display: none;
        }
        
        .form-container {
            background: rgba(255, 255, 255, 0.85);
            backdrop-filter: blur(16px);
            box-shadow: 0 30px 60px -40px rgba(15, 23, 42, 0.5);
            border: 1px solid rgba(148, 163, 184, 0.35);
            border-radius: 24px;
        }

        .premium-input {
            background: rgba(255,255,255,0.85);
            border: 1px solid rgba(148,163,184,0.35);
            box-shadow: 0 10px 18px -16px rgba(15,23,42,0.45);
        }

        .premium-card {
            border: 1px solid rgba(148, 163, 184, 0.35);
            background: #ffffff;
            box-shadow: 0 18px 36px -28px rgba(15, 23, 42, 0.45);
        }

        .form-container input,
        .form-container select,
        .form-container textarea {
            background: rgba(255, 255, 255, 0.85);
            border: 1px solid rgba(148, 163, 184, 0.35);
            box-shadow: 0 10px 18px -16px rgba(15, 23, 42, 0.45);
        }

        .form-container input:focus,
        .form-container select:focus,
        .form-container textarea:focus {
            outline: none;
            border-color: rgba(14, 165, 168, 0.6);
            box-shadow: 0 12px 24px -16px rgba(14, 116, 144, 0.45);
        }

        .hospital-option,
        .speciality-option,
        .doctor-option,
        .time-slot {
            background: #ffffff;
            border: 1px solid rgba(148, 163, 184, 0.35);
            box-shadow: 0 14px 28px -24px rgba(15, 23, 42, 0.4);
        }

        .hospital-option:hover,
        .speciality-option:hover,
        .doctor-option:hover,
        .time-slot:hover {
            border-color: rgba(14, 165, 168, 0.5);
        }
        
        .speciality-option, .doctor-option {
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            transform: translateY(0);
        }
        
        .speciality-option:hover, .doctor-option:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 30px rgba(14, 165, 233, 0.15);
        }
        
        .step-indicator {
            transition: all 0.3s ease;
        }
        
        .step-indicator:hover {
            transform: scale(1.05);
        }
        
        .time-slot {
            transition: all 0.3s ease;
        }
        
        .time-slot:hover {
            transform: translateY(-3px);
            box-shadow: 0 10px 20px rgba(14, 165, 233, 0.2);
        }
        
        .form-step {
            transition: opacity 0.4s ease, transform 0.4s ease;
        }
        
        .form-step.active {
            opacity: 1;
            transform: translateY(0);
        }
        
        .form-step:not(.active) {
            opacity: 0;
            transform: translateY(20px);
            display: none;
        }
        
        .email-availability-message {
            margin-top: 5px;
            font-size: 0.875rem;
            padding: 4px 8px;
            border-radius: 4px;
            display: none;
        }
        
        .email-available {
            background-color: #d1fae5;
            color: #065f46;
            border: 1px solid #10b981;
        }
        
        .email-taken {
            background-color: #fee2e2;
            color: #991b1b;
            border: 1px solid #ef4444;
        }
        
        .email-invalid {
            background-color: #fef3c7;
            color: #92400e;
            border: 1px solid #f59e0b;
        }
    </style>
</head>

<body class="font-sans antialiased min-h-screen">
    <!-- Navbar -->
    <nav class="bg-white/80 backdrop-blur-md shadow-custom sticky top-0 z-50 border-b border-gray-200/50">
        <div class="max-w-7xl mx-auto px-6">
            <div class="flex justify-between items-center h-20">
                <div class="flex-shrink-0">
                    <a href="<?php echo base_url(); ?>frontend" class="flex items-center space-x-3">
                        <div class="relative w-10 h-10">
                            <div class="absolute inset-0 bg-gradient-to-r from-primary-600 to-primary-500 rounded-lg transform rotate-6"></div>
                            <div class="absolute inset-0 bg-white rounded-lg shadow-sm flex items-center justify-center">
                                <span class="text-xl font-bold text-primary-600">
                                    <?php echo substr($settings->title, 0, 1); ?>
                                </span>
                            </div>
                        </div>
                        <div>
                            <span class="text-2xl font-extrabold">
                                <span class="text-secondary-900"><?php echo $title[0]; ?></span><span class="text-primary-600"><?php echo isset($title[1]) ? $title[1] : ''; ?></span>
                            </span>
                        </div>
                    </a>
                </div>
                
                <div class="hidden md:flex items-center space-x-6">
                    <a href="<?php echo base_url(); ?>frontend" class="text-gray-700 hover:text-primary-600 transition-colors duration-200 font-medium">
                        <i class="fas fa-arrow-left mr-2"></i>Back to Home
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Main Content -->
    <div class="container mx-auto px-4 py-12">
        <div class="max-w-4xl mx-auto">
            <!-- Header -->
            <div class="text-center mb-12">
                <h1 class="text-4xl md:text-5xl font-bold text-gray-900 mb-4 serif-head">
                    Book Your <span class="gradient-text">Medical Appointment</span>
                </h1>
                <p class="text-xl text-gray-600 max-w-2xl mx-auto">
                    Find the right doctor, choose your preferred time, and get the care you need
                </p>
            </div>

            <!-- Progress Steps -->
            <div class="flex justify-between mb-10 relative">
                <div class="absolute top-4 left-0 right-0 h-1 bg-gray-200 z-0"></div>
                <div class="absolute top-4 left-0 h-1 bg-gradient-to-r from-primary-500 to-primary-600 z-10 transition-all duration-500" id="progress-bar" style="width: 20%"></div>
                
                <div class="relative z-20 flex justify-between w-full flex-wrap">
                    <div class=" flex flex-col items-center min-w-[60px]">
                        <div class="step-indicator w-12 h-12 rounded-full bg-gradient-to-r from-primary-500 to-primary-600 text-white flex items-center justify-center mb-2 shadow-lg">
                            <i class="fas fa-hospital"></i>
                        </div>
                        <span class="text-sm font-medium text-gray-700 text-center">Hospital</span>
                    </div>
                    
                    <div class=" flex flex-col items-center min-w-[60px]">
                        <div class="step-indicator w-12 h-12 rounded-full bg-white border-2 border-gray-300 text-gray-400 flex items-center justify-center mb-2 shadow transition-all">
                            <i class="fas fa-stethoscope"></i>
                        </div>
                        <span class="text-sm font-medium text-gray-500 text-center">Speciality</span>
                    </div>
                    
                    <div class=" flex flex-col items-center min-w-[60px]">
                        <div class="step-indicator w-12 h-12 rounded-full bg-white border-2 border-gray-300 text-gray-400 flex items-center justify-center mb-2 shadow transition-all">
                            <i class="fas fa-user-md"></i>
                        </div>
                        <span class="text-sm font-medium text-gray-500 text-center">Doctor</span>
                    </div>
                    
                    <div class=" flex flex-col items-center min-w-[60px]">
                        <div class="step-indicator w-12 h-12 rounded-full bg-white border-2 border-gray-300 text-gray-400 flex items-center justify-center mb-2 shadow transition-all">
                            <i class="fas fa-calendar"></i>
                        </div>
                        <span class="text-sm font-medium text-gray-500 text-center">Date</span>
                    </div>
                    
                    <div class=" flex flex-col items-center min-w-[60px]">
                        <div class="step-indicator w-12 h-12 rounded-full bg-white border-2 border-gray-300 text-gray-400 flex items-center justify-center mb-2 shadow transition-all">
                            <i class="fas fa-clock"></i>
                        </div>
                        <span class="text-sm font-medium text-gray-500 text-center">Time</span>
                    </div>
                </div>
            </div>

            <!-- Booking Form -->
            <div class="form-container rounded-2xl shadow-xl p-8">
                <form id="appointment-form" class="space-y-8">
                    <!-- Step 1: Hospital Selection -->
                    <div class="form-step active" id="step-1">
                        <h2 class="text-2xl font-bold text-gray-900 mb-6 flex items-center">
                            <i class="fas fa-hospital text-primary-600 mr-3"></i>
                            Select Hospital
                        </h2>
                        
                        <!-- Hospital Search -->
                        <div class="mb-6">
                            <div class="relative">
                                <input 
                                    type="text" 
                                    id="hospital-search" 
                                    placeholder="Search hospitals by name, location, or specialty..."
                                    class="w-full px-4 py-3 pl-12 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent transition-all"
                                >
                                <i class="fas fa-search absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400"></i>
                            </div>
                        </div>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4" id="hospitals-container">
                            <?php foreach($hospitals as $hospital): ?>
                            <div class="hospital-option border-2 border-gray-200 rounded-xl p-6 cursor-pointer hover:border-primary-300 transition-all duration-300 hover:shadow-xl group" 
                                 data-hospital-id="<?php echo $hospital->id; ?>">
                                <div class="flex items-start space-x-4">
                                    <div class="flex-shrink-0">
                                        <div class="w-14 h-14 bg-gradient-to-r from-primary-500 to-primary-600 rounded-xl flex items-center justify-center group-hover:scale-105 transition-transform duration-300 shadow-md">
                                            <i class="fas fa-hospital text-white text-xl"></i>
                                        </div>
                                    </div>
                                    <div class="flex-1">
                                        <h3 class="text-lg font-bold text-gray-900 group-hover:text-primary-600 transition-colors mb-2">
                                            <?php echo htmlspecialchars($hospital->name); ?>
                                        </h3>
                                        <p class="text-gray-600 text-sm mt-1 flex items-center mb-1">
                                            <i class="fas fa-map-marker-alt text-xs mr-2 text-primary-500"></i>
                                            <?php echo htmlspecialchars($hospital->address); ?>
                                        </p>
                                        <p class="text-gray-500 text-sm flex items-center">
                                            <i class="fas fa-phone text-xs mr-2 text-primary-500"></i>
                                            <?php echo htmlspecialchars($hospital->phone); ?>
                                        </p>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <input type="hidden" name="hospital_id" id="hospital_id" required>
                    </div>

                    <!-- Step 2: Speciality Selection -->
                    <div class="form-step hidden" id="step-2">
                        <h2 class="text-2xl font-bold text-gray-900 mb-6">Select Department</h2>
                        <div class="mb-6">
                            <div class="relative">
                                <input 
                                    type="text" 
                                    id="speciality-search" 
                                    placeholder="Search departments..."
                                    class="w-full px-4 py-3 pl-12 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent transition-all"
                                >
                                <i class="fas fa-search absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400"></i>
                            </div>
                        </div>
                        <div class="grid grid-cols-2 md:grid-cols-3 gap-4" id="specialities-container">
                            <!-- Specialities will be loaded here -->
                            <div class="speciality-option border-2 border-gray-200 rounded-xl p-6 cursor-pointer hover:border-primary-300 transition-all duration-300 text-center shadow-sm hover:shadow-md opacity-0" style="visibility:hidden;">
                                <div class="w-14 h-14 bg-gradient-to-r from-primary-100 to-primary-200 rounded-full flex items-center justify-center mx-auto mb-3 transition-colors">
                                    <i class="fas fa-stethoscope text-2xl text-primary-600"></i>
                                </div>
                                <div class="font-semibold text-gray-800">Sample</div>
                            </div>
                        </div>
                        <input type="hidden" name="speciality" id="speciality" required>
                    </div>

                    <!-- Step 3: Doctor Selection -->
                    <div class="form-step hidden" id="step-3">
                        <h2 class="text-2xl font-bold text-gray-900 mb-6">Select Doctor</h2>
                        <div class="mb-6">
                            <div class="relative">
                                <input 
                                    type="text" 
                                    id="doctor-search" 
                                    placeholder="Search doctors..."
                                    class="w-full px-4 py-3 pl-12 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent transition-all"
                                >
                                <i class="fas fa-search absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400"></i>
                            </div>
                        </div>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6" id="doctors-container">
                            <!-- Doctors will be loaded here -->
                            <div class="doctor-option border-2 border-gray-200 rounded-xl p-6 cursor-pointer hover:border-primary-300 transition-all duration-300 hover:shadow-lg opacity-0" style="visibility:hidden;">
                                <div class="flex items-start space-x-4">
                                    <div class="flex-shrink-0">
                                        <div class="w-14 h-14 bg-gradient-to-r from-primary-100 to-primary-200 rounded-full flex items-center justify-center shadow-sm">
                                            <i class="fas fa-user-md text-primary-600 text-lg"></i>
                                        </div>
                                    </div>
                                    <div class="flex-1">
                                        <h3 class="text-lg font-bold text-gray-900 mb-1">Dr. Sample</h3>
                                        <p class="text-primary-600 text-sm font-medium mb-1">Department</p>
                                        <p class="text-gray-600 text-sm">Sample doctor profile information...</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <input type="hidden" name="doctor_id" id="doctor_id" required>
                    </div>

                    <!-- Step 4: Date Selection -->
                    <div class="form-step hidden" id="step-4">
                        <h2 class="text-2xl font-bold text-gray-900 mb-6">Select Date & Visit Details</h2>
                        <div class="mb-6">
                            <label class="block text-lg font-medium text-gray-700 mb-3">Visit Type</label>
                            <select class="w-full px-4 py-3 border-2 border-gray-200 rounded-lg focus:ring-2 focus:ring-primary-300 focus:border-primary-500 transition-colors shadow-sm" name="visit_description" id="visit_description" required>
                                <option value="">Select Visit Type</option>
                            </select>
                        </div>
                        
                        <?php if ($settings1->show_tawkto_in_website == 'yes'): ?>
                        <div class="mb-6" id="visit-amount-section">
                            <label class="block text-lg font-medium text-gray-700 mb-3">Visit Amount</label>
                            <input type="text" class="w-full px-4 py-3 border-2 border-gray-200 rounded-lg focus:ring-2 focus:ring-primary-300 focus:border-primary-500 transition-colors shadow-sm" name="visit_charges" id="visit_charges" placeholder="Visit amount will be filled automatically" readonly>
                            <div id="visit-charges-message" class="email-availability-message mt-2"></div>
                        </div>
                        <?php else: ?>
                        <div style="display: none;" id="visit-hidden-amount-section">
                            <input type="hidden" name="visit_charges" id="visit_charges_hidden" />
                        </div>
                        <?php endif; ?>
                        <div class="mb-6">
                            <label class="block text-lg font-medium text-gray-700 mb-3">Select Date</label>
                            <div class="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 gap-4" id="dates-container">
                                <!-- Dates will be loaded here -->
                            </div>
                        </div>
                        <input type="hidden" name="appointment_date" id="appointment_date" required>
                    </div>

                    <!-- Step 5: Time Slot Selection -->
                    <div class="form-step hidden" id="step-5">
                        <h2 class="text-2xl font-bold text-gray-900 mb-6">Select Time Slot</h2>
                        <div class="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 gap-4" id="timeslots-container">
                            <!-- Time slots will be loaded here -->
                        </div>
                        <input type="hidden" name="time_slot" id="time_slot" required>
                    </div>

                    <!-- Step 6: Patient Information -->
                    <div class="form-step hidden" id="step-6">
                        <h2 class="text-2xl font-bold text-gray-900 mb-6">Patient Information</h2>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Full Name *</label>
                                <input type="text" name="patient_name" id="patient_name" required
                                       class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-all shadow-sm">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Email Address *</label>
                                <input type="email" name="patient_email" id="patient_email" required
                                       class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-all shadow-sm">
                                <div id="email-availability-message" class="email-availability-message"></div>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Phone Number *</label>
                                <input type="tel" name="patient_phone" id="patient_phone" required
                                       class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-all shadow-sm">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Address</label>
                                <input type="text" name="patient_address" id="patient_address"
                                       class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-all shadow-sm">
                            </div>
                            <div class="md:col-span-2">
                                <label class="block text-sm font-medium text-gray-700 mb-2">Visit Description/Reason</label>
                                <textarea name="visit_description" id="visit_description" rows="4"
                                          class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-all shadow-sm"
                                          placeholder="Please describe the reason for your visit..."></textarea>
                            </div>
                        </div>
                    </div>

                    <!-- Navigation Buttons -->
                    <div class="flex justify-between pt-8 border-t border-gray-200">
                        <button type="button" id="prev-btn" class="px-6 py-3 border border-primary-300 text-primary-600 rounded-lg hover:bg-primary-50 transition-all duration-300 flex items-center">
                            <i class="fas fa-arrow-left mr-2"></i>Previous
                        </button>
                        <div class="flex gap-3">
                            <button type="submit" id="submit-btn" class="px-6 py-3 bg-gradient-to-r from-green-500 to-green-600 text-white rounded-lg hover:from-green-600 hover:to-green-700 transition-all duration-300 flex items-center shadow-md hidden">
                                <i class="fas fa-calendar-check mr-2"></i>Confirm Appointment
                            </button>
                            <button type="button" id="next-btn" class="px-6 py-3 bg-gradient-to-r from-primary-500 to-primary-600 text-white rounded-lg hover:from-primary-600 hover:to-primary-700 transition-all duration-300 flex items-center shadow-md">
                                Next<i class="fas fa-arrow-right ml-2"></i>
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Loading Spinner -->
    <div id="loading-overlay" class="fixed inset-0 bg-black/50 flex items-center justify-center z-50 hidden">
        <div class="bg-white rounded-xl p-8 text-center">
            <div class="animate-spin rounded-full h-12 w-12 border-b-2 border-primary-600 mx-auto mb-4"></div>
            <p class="text-gray-700">Processing your appointment...</p>
        </div>
    </div>

    <!-- Success Modal -->
    <div id="success-modal" class="fixed inset-0 bg-black/50 flex items-center justify-center z-50 hidden">
        <div class="bg-white rounded-2xl p-8 max-w-md w-full mx-4">
            <div class="text-center">
                <div class="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                    <i class="fas fa-check text-2xl text-blue-600"></i>
                </div>
                <h3 class="text-2xl font-bold text-gray-900 mb-2">Appointment Booked!</h3>
                <p class="text-gray-600 mb-6" id="success-message"></p>
                <button id="close-success-modal" class="px-6 py-3 bg-primary-600 text-white rounded-lg hover:bg-primary-700 transition-colors">
                    Continue
                </button>
            </div>
        </div>
    </div>

    <!-- Error Modal -->
    <div id="error-modal" class="fixed inset-0 bg-black/50 flex items-center justify-center z-50 hidden">
        <div class="bg-white rounded-2xl p-8 max-w-md w-full mx-4">
            <div class="text-center">
                <div class="w-16 h-16 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-4">
                    <i class="fas fa-exclamation-triangle text-2xl text-red-600"></i>
                </div>
                <h3 class="text-2xl font-bold text-gray-900 mb-2">Alert</h3>
                <p class="text-gray-600 mb-6" id="error-message"></p>
                <button id="close-error-modal" class="px-6 py-3 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors">
                    Try Again
                </button>
            </div>
        </div>
    </div>

    <script>
        let currentStep = 1;
        const totalSteps = 6;
        
        $(document).ready(function() {
            // Hospital search functionality
            $('#hospital-search').on('input', function() {
                const searchTerm = $(this).val().toLowerCase();
                
                $('.hospital-option').each(function() {
                    const hospitalText = $(this).find('h3').text().toLowerCase() + 
                                $(this).find('p').text().toLowerCase();
                    
                    if (hospitalText.includes(searchTerm)) {
                        $(this).show();
                    } else {
                        $(this).hide();
                    }
                });
            });
            
            // Hospital selection
            $(document).on('click', '.hospital-option', function() {
                $('.hospital-option').removeClass('border-primary-500 bg-primary-50').addClass('border-gray-200');
                $(this).removeClass('border-gray-200').addClass('border-primary-500 bg-primary-50');
                $('#hospital_id').val($(this).data('hospital-id'));
                
                // Update step indicator to show hospital as completed
                updateStepIndicator(1, 'completed');
                
                // Move to step 2
                goToStep(2);
                
                // Load specialities for selected hospital
                loadSpecialities($(this).data('hospital-id'));
                
                // Clear specialty search when hospital changes
                $('#speciality-search').val('');
            });
            
            // Speciality selection
            $(document).on('click', '.speciality-option', function() {
                $('.speciality-option').removeClass('border-primary-500 bg-primary-50').addClass('border-gray-200');
                $(this).removeClass('border-gray-200').addClass('border-primary-500 bg-primary-50');
                $('#speciality').val($(this).data('speciality'));
                
                // Update step indicator to show specialty as completed
                updateStepIndicator(2, 'completed');
                
                // Move to step 3
                goToStep(3);
                
                // Load doctors for selected speciality
                loadDoctors($('#hospital_id').val(), $(this).data('speciality'));
                
                // Clear doctor search when specialty changes
                $('#doctor-search').val('');
            });
            
            // Doctor selection
            $(document).on('click', '.doctor-option', function() {
                $('.doctor-option').removeClass('border-primary-500 bg-primary-50').addClass('border-gray-200');
                $(this).removeClass('border-gray-200').addClass('border-primary-500 bg-primary-50');
                $('#doctor_id').val($(this).data('doctor-id'));
                
                // Update step indicator to show doctor as completed
                updateStepIndicator(3, 'completed');
                
                // Move to step 4
                goToStep(4);
                
                // Load visit types for the selected doctor
                const doctorId = $(this).data('doctor-id');
                if (doctorId) {
                    loadVisitTypes(doctorId);
                }
                
                // Load available dates for selected doctor
                loadAvailableDates($(this).data('doctor-id'));
            });
            
            // Specialty search functionality
            let specialitySearchTimeout;
            $('#speciality-search').on('keyup', function() {
                const searchValue = $(this).val().trim();
                
                // Clear previous timeout
                clearTimeout(specialitySearchTimeout);
                
                if (searchValue.length === 0) {
                    // If search is cleared, reload all specialities
                    const hospitalId = $('#hospital_id').val();
                    if (hospitalId) {
                        loadSpecialities(hospitalId);
                    }
                    return;
                }
                
                if (searchValue.length >= 2) {
                    specialitySearchTimeout = setTimeout(function() {
                        const hospitalId = $('#hospital_id').val();
                        if (hospitalId) {
                            searchSpecialities(hospitalId, searchValue);
                        }
                    }, 500);
                }
            });
            
            // Doctor search functionality
            let doctorSearchTimeout;
            $('#doctor-search').on('keyup', function() {
                const searchValue = $(this).val().trim();
                
                // Clear previous timeout
                clearTimeout(doctorSearchTimeout);
                
                if (searchValue.length === 0) {
                    // If search is cleared, reload all doctors
                    const hospitalId = $('#hospital_id').val();
                    const speciality = $('#speciality').val();
                    if (hospitalId && speciality) {
                        loadDoctors(hospitalId, speciality);
                    }
                    return;
                }
                
                if (searchValue.length >= 2) {
                    doctorSearchTimeout = setTimeout(function() {
                        const hospitalId = $('#hospital_id').val();
                        const speciality = $('#speciality').val();
                        if (hospitalId && speciality) {
                            searchDoctors(hospitalId, speciality, searchValue);
                        }
                    }, 500);
                }
            });
            
            // Date selection
            $(document).on('click', '.date-option', function() {
                $('.date-option').removeClass('border-primary-500 bg-primary-50 text-primary-700').addClass('border-gray-200 text-gray-700');
                $(this).removeClass('border-gray-200 text-gray-700').addClass('border-primary-500 bg-primary-50 text-primary-700');
                $('#appointment_date').val($(this).data('date'));
                
                // Update step indicator to show date as completed
                updateStepIndicator(4, 'completed');
                
                // Move to step 5
                goToStep(5);
                
                // Load time slots for selected date
                loadTimeSlots($('#doctor_id').val(), $(this).data('date'));
            });
            
            // Time slot selection
            $(document).on('click', '.time-slot', function() {
                $('.time-slot').removeClass('selected border-primary-500 bg-primary-500 text-white').addClass('border-gray-200 text-gray-700');
                $(this).removeClass('border-gray-200 text-gray-700').addClass('selected border-primary-500 bg-primary-500 text-white');
                $('#time_slot').val($(this).data('time'));
                
                // Update step indicator to show time as completed
                updateStepIndicator(5, 'completed');
                
                // Move to step 6
                goToStep(6);
            });
            
            // Visit description selection
            $(document).on('change', '#visit_description', function() {
                const visitTypeId = $(this).val();
                if (visitTypeId) {
                    loadVisitCharges(visitTypeId);
                } else {
                    $('#visit_charges').val('');
                    $('#visit_charges_hidden').val('');
                }
            });
            
            // Navigation
            $('#next-btn').click(function() {
                if (validateStep(currentStep)) {
                    goToStep(currentStep + 1);
                }
            });
            
            $('#prev-btn').click(function() {
                goToStep(currentStep - 1);
            });
            
            // Form submission
            $('#appointment-form').submit(function(e) {
                e.preventDefault();
                submitAppointment();
            });
            
            // Modal close buttons
            $('#close-success-modal').click(function() {
                $('#success-modal').addClass('hidden');
                window.location.href = '<?php echo base_url(); ?>frontend';
            });
            
            $('#close-error-modal').click(function() {
                $('#error-modal').addClass('hidden');
            });
            
            // Initialize buttons state
            $('#prev-btn').addClass('hidden'); // Hide previous button on initial step
            
            // Email availability checking
            let emailCheckTimeout;
            $('#patient_email').on('keyup', function() {
                const email = $(this).val().trim();
                const emailMessage = $('#email-availability-message');
                
                // Clear previous timeout
                clearTimeout(emailCheckTimeout);
                
                // Hide message initially
                emailMessage.hide().removeClass('email-available email-taken email-invalid');
                
                // Validate email format first
                if (email === '') {
                    return;
                }
                
                if (!isValidEmail(email)) {
                    emailMessage.text('Please enter a valid email address').addClass('email-invalid').show();
                    return;
                }
                
                // Show checking message
                emailMessage.text('Checking availability...').addClass('email-invalid').show();
                
                // Delay the check to avoid too many requests
                emailCheckTimeout = setTimeout(function() {
                    checkEmailAvailability(email);
                }, 500);
            });
            
            function isValidEmail(email) {
                const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                return emailRegex.test(email);
            }
            
            function checkEmailAvailability(email) {
                $.ajax({
                    url: '<?php echo base_url(); ?>frontend/checkEmailAvailability',
                    method: 'POST',
                    data: { email: email },
                    dataType: 'json',
                    timeout: 10000,
                    success: function(response) {
                        const emailMessage = $('#email-availability-message');
                        emailMessage.removeClass('email-available email-taken email-invalid');
                        
                        if (response.status === 'available') {
                            emailMessage.text(response.message).addClass('email-available').show();
                        } else if (response.status === 'taken') {
                            emailMessage.text(response.message).addClass('email-taken').show();
                        } else {
                            emailMessage.text(response.message).addClass('email-invalid').show();
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('Email check error:', xhr, status, error);
                        const emailMessage = $('#email-availability-message');
                        emailMessage.removeClass('email-available email-taken').addClass('email-invalid').text('Error checking email availability').show();
                    }
                });
            }
        });
        
        function loadSpecialities(hospitalId) {
            $.ajax({
                url: '<?php echo base_url(); ?>frontend/get_doctor_specialities',
                method: 'POST',
                data: { hospital_id: hospitalId },
                dataType: 'json',
                timeout: 10000, // 10 second timeout
                beforeSend: function() {
                    showLoading();
                },
                success: function(response) {
                    hideLoading();
                    if (response && response.status === 'success' && response.specialities) {
                        let html = '';
                        response.specialities.forEach(function(speciality) {
                            if (speciality.speciality) {
                                // Use ID if available, otherwise use the speciality name
                                const specialityId = speciality.id || speciality.speciality;
                                html += `
                                    <div class="speciality-option border-2 border-gray-200 rounded-lg p-4 cursor-pointer hover:border-primary-300 transition-colors duration-200 text-center" 
                                         data-speciality="${specialityId}">
                                        <i class="fas fa-stethoscope text-2xl text-primary-600 mb-2"></i>
                                        <div class="font-medium text-gray-900">${speciality.speciality}</div>
                                    </div>
                                `;
                            }
                        });
                        if (html) {
                            $('#specialities-container').html(html);
                            goToStep(2); // Move to step 2
                        } else {
                            showError('No specialities found for this hospital. Please select another hospital.');
                        }
                    } else {
                        showError(response.message || 'Failed to load specialities. Please try again.');
                    }
                },
                error: function(xhr, status, error) {
                    hideLoading();
                    console.error('AJAX Error:', xhr, status, error);
                    if (status === 'timeout') {
                        showError('Request timed out. Please check your internet connection and try again.');
                    } else if (xhr.status === 404) {
                        showError('Server endpoint not found. Please contact support.');
                    } else {
                        showError('Network error occurred. Please check your internet connection and try again.');
                    }
                }
            });
        }
        
        function loadDoctors(hospitalId, speciality) {
            $.ajax({
                url: '<?php echo base_url(); ?>frontend/get_doctors_by_speciality',
                method: 'POST',
                data: { 
                    hospital_id: hospitalId,
                    speciality: speciality
                },
                dataType: 'json',
                timeout: 10000, // 10 second timeout
                beforeSend: function() {
                    showLoading();
                },
                success: function(response) {
                    hideLoading();
                    if (response && response.status === 'success' && response.doctors) {
                        let html = '';
                        response.doctors.forEach(function(doctor) {
                            if (doctor.id && doctor.name) {
                                html += `
                                    <div class="doctor-option border-2 border-gray-200 rounded-xl p-6 cursor-pointer hover:border-primary-300 transition-colors duration-200" 
                                         data-doctor-id="${doctor.id}">
                                        <div class="flex items-center space-x-4">
                                            <div class="flex-shrink-0">
                                                <div class="w-12 h-12 bg-primary-100 rounded-full flex items-center justify-center">
                                                    <i class="fas fa-user-md text-primary-600"></i>
                                                </div>
                                            </div>
                                            <div class="flex-1">
                                                <h3 class="text-lg font-semibold text-gray-900">${doctor.name}</h3>
                                                <p class="text-gray-600 text-sm">${doctor.department_name}</p>
                                                ${doctor.profile ? `<p class="text-gray-500 text-xs mt-1">${doctor.profile.substring(0, 100)}...</p>` : ''}
                                            </div>
                                        </div>
                                    </div>
                                `;
                            }
                        });
                        if (html) {
                            $('#doctors-container').html(html);
                            goToStep(3); // Move to step 3
                        } else {
                            showError('No doctors found for this speciality. Please select another speciality.');
                        }
                    } else {
                        showError(response.message || 'Failed to load doctors. Please try again.');
                    }
                },
                error: function(xhr, status, error) {
                    hideLoading();
                    console.error('AJAX Error:', xhr, status, error);
                    if (status === 'timeout') {
                        showError('Request timed out. Please check your internet connection and try again.');
                    } else if (xhr.status === 404) {
                        showError('Server endpoint not found. Please contact support.');
                    } else {
                        showError('Network error occurred. Please check your internet connection and try again.');
                    }
                }
            });
        }
        
        function loadAvailableDates(doctorId) {
            $.ajax({
                url: '<?php echo base_url(); ?>frontend/get_available_dates',
                method: 'POST',
                data: { doctor_id: doctorId },
                dataType: 'json',
                timeout: 10000, // 10 second timeout
                beforeSend: function() {
                    showLoading();
                },
                success: function(response) {
                    hideLoading();
                    if (response && response.status === 'success' && response.dates) {
                        let html = '';
                        response.dates.forEach(function(date) {
                            if (date.date) {
                                html += `
                                    <div class="date-option border-2 border-gray-200 rounded-lg p-4 cursor-pointer hover:border-primary-300 transition-colors duration-200 text-center" 
                                         data-date="${date.date}">
                                        <div class="font-medium text-gray-900">${date.display_date}</div>
                                        <div class="text-sm text-gray-500">${date.weekday}</div>
                                    </div>
                                `;
                            }
                        });
                        if (html) {
                            $('#dates-container').html(html);
                            goToStep(4); // Move to step 4
                        } else {
                            showError('No available dates found for this doctor. Please select another doctor.');
                        }
                    } else {
                        showError(response.message || 'Failed to load available dates. Please try again.');
                    }
                },
                error: function(xhr, status, error) {
                    hideLoading();
                    console.error('AJAX Error:', xhr, status, error);
                    if (status === 'timeout') {
                        showError('Request timed out. Please check your internet connection and try again.');
                    } else if (xhr.status === 404) {
                        showError('Server endpoint not found. Please contact support.');
                    } else {
                        showError('Network error occurred. Please check your internet connection and try again.');
                    }
                }
            });
        }
        
        function loadTimeSlots(doctorId, date) {
            $.ajax({
                url: '<?php echo base_url(); ?>frontend/get_available_timeslots',
                method: 'POST',
                data: { 
                    doctor_id: doctorId,
                    date: date
                },
                dataType: 'json',
                timeout: 10000, // 10 second timeout
                beforeSend: function() {
                    showLoading();
                },
                success: function(response) {
                    hideLoading();
                    if (response && response.status === 'success' && response.timeslots) {
                        let html = '';
                        response.timeslots.forEach(function(slot) {
                            if (slot.time) {
                                html += `
                                    <div class="time-slot border-2 border-gray-200 rounded-lg p-4 cursor-pointer hover:border-primary-300 transition-all duration-200 text-center" 
                                         data-time="${slot.time}">
                                        <div class="font-medium text-gray-700">${slot.time}</div>
                                    </div>
                                `;
                            }
                        });
                        if (html) {
                            $('#timeslots-container').html(html);
                            goToStep(5); // Move to step 5
                        } else {
                            showError('No available time slots for this date. Please select another date.');
                        }
                    } else {
                        showError(response.message || 'Failed to load time slots. Please try again.');
                    }
                },
                error: function(xhr, status, error) {
                    hideLoading();
                    console.error('AJAX Error:', xhr, status, error);
                    if (status === 'timeout') {
                        showError('Request timed out. Please check your internet connection and try again.');
                    } else if (xhr.status === 404) {
                        showError('Server endpoint not found. Please contact support.');
                    } else {
                        showError('Network error occurred. Please check your internet connection and try again.');
                    }
                }
            });
        }
        
        function loadVisitTypes(doctorId) {
            $.ajax({
                url: '<?php echo base_url(); ?>frontend/getDoctorVisit?id=' + doctorId,
                method: 'GET',
                dataType: 'json',
                timeout: 10000, // 10 second timeout
                beforeSend: function() {
                    showLoading();
                },
                success: function(response) {
                    hideLoading();
                    if (response && response.response) {
                        $('#visit_description').html(response.response);
                    } else {
                        showError('Failed to load visit types. Please try again.');
                    }
                },
                error: function(xhr, status, error) {
                    hideLoading();
                    console.error('AJAX Error:', xhr, status, error);
                    if (status === 'timeout') {
                        showError('Request timed out. Please check your internet connection and try again.');
                    } else if (xhr.status === 404) {
                        showError('Server endpoint not found. Please contact support.');
                    } else {
                        showError('Network error occurred. Please check your internet connection and try again.');
                    }
                }
            });
        }
        
        function searchSpecialities(hospitalId, searchValue) {
            $.ajax({
                url: '<?php echo base_url(); ?>frontend/searchSpecialities',
                method: 'POST',
                data: { 
                    hospital_id: hospitalId,
                    search_term: searchValue
                },
                dataType: 'json',
                timeout: 10000, // 10 second timeout
                success: function(response) {
                    if (response && response.status === 'success' && response.specialities) {
                        let html = '';
                        response.specialities.forEach(function(speciality) {
                            if (speciality.speciality) {
                                // Use ID if available, otherwise use the speciality name
                                const specialityId = speciality.id || speciality.speciality;
                                html += `
                                    <div class="speciality-option border-2 border-gray-200 rounded-xl p-6 cursor-pointer hover:border-primary-300 transition-all duration-300 text-center shadow-sm hover:shadow-md" 
                                         data-speciality="${specialityId}">
                                    <div class="w-14 h-14 bg-gradient-to-r from-primary-100 to-primary-200 rounded-full flex items-center justify-center mx-auto mb-3 transition-colors">
                                        <i class="fas fa-stethoscope text-2xl text-primary-600"></i>
                                    </div>
                                    <div class="font-semibold text-gray-800">${speciality.speciality}</div>
                                    </div>
                                `;
                            }
                        });
                        if (html) {
                            $('#specialities-container').html(html);
                        } else {
                            $('#specialities-container').html(`<div class="text-center text-gray-500 py-8 col-span-full">No specialities found for your search.</div>`);
                        }
                    } else {
                        showError(response.message || 'Failed to search specialities. Please try again.');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX Error:', xhr, status, error);
                    if (status === 'timeout') {
                        showError('Request timed out. Please check your internet connection and try again.');
                    } else if (xhr.status === 404) {
                        showError('Server endpoint not found. Please contact support.');
                    } else {
                        showError('Network error occurred. Please check your internet connection and try again.');
                    }
                }
            });
        }
        
        function searchDoctors(hospitalId, speciality, searchValue) {
            $.ajax({
                url: '<?php echo base_url(); ?>frontend/searchDoctors',
                method: 'POST',
                data: { 
                    hospital_id: hospitalId,
                    speciality: speciality,
                    search_term: searchValue
                },
                dataType: 'json',
                timeout: 10000, // 10 second timeout
                success: function(response) {
                    if (response && response.status === 'success' && response.doctors) {
                        let html = '';
                        response.doctors.forEach(function(doctor) {
                            if (doctor.id && doctor.name) {
                                html += `
                                    <div class="doctor-option border-2 border-gray-200 rounded-xl p-6 cursor-pointer hover:border-primary-300 transition-all duration-300 hover:shadow-lg" 
                                         data-doctor-id="${doctor.id}">
                                        <div class="flex items-start space-x-4">
                                            <div class="flex-shrink-0">
                                                <div class="w-14 h-14 bg-gradient-to-r from-primary-100 to-primary-200 rounded-full flex items-center justify-center shadow-sm">
                                                    <i class="fas fa-user-md text-primary-600 text-lg"></i>
                                                </div>
                                            </div>
                                            <div class="flex-1">
                                                <h3 class="text-lg font-bold text-gray-900 mb-1">${doctor.name}</h3>
                                                <p class="text-primary-600 text-sm font-medium mb-1">${doctor.department_name}</p>
                                                ${doctor.profile ? `<p class="text-gray-600 text-sm">${doctor.profile.substring(0, 100)}...</p>` : ''}
                                            </div>
                                        </div>
                                    </div>
                                `;
                            }
                        });
                        if (html) {
                            $('#doctors-container').html(html);
                        } else {
                            $('#doctors-container').html(`<div class="text-center text-gray-500 py-8 col-span-full">No doctors found for your search.</div>`);
                        }
                    } else {
                        showError(response.message || 'Failed to search doctors. Please try again.');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX Error:', xhr, status, error);
                    if (status === 'timeout') {
                        showError('Request timed out. Please check your internet connection and try again.');
                    } else if (xhr.status === 404) {
                        showError('Server endpoint not found. Please contact support.');
                    } else {
                        showError('Network error occurred. Please check your internet connection and try again.');
                    }
                }
            });
        }
        
        function loadVisitCharges(visitTypeId) {
            $.ajax({
                url: '<?php echo base_url(); ?>frontend/getDoctorVisitCharges?id=' + visitTypeId,
                method: 'GET',
                dataType: 'json',
                timeout: 10000, // 10 second timeout
                success: function(response) {
                    if (response && response.response) {
                        $('#visit_charges').val(response.response.visit_charges);
                        $('#visit_charges_hidden').val(response.response.visit_charges); // Also update hidden field
                    } else {
                        $('#visit_charges').val('');
                        $('#visit_charges_hidden').val(''); // Also update hidden field
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Error loading visit charges:', xhr, status, error);
                    $('#visit_charges').val('');
                    $('#visit_charges_hidden').val(''); // Also update hidden field
                }
            });
        }
        
        function validateStep(step) {
            switch(step) {
                case 1:
                    if (!$('#hospital_id').val()) {
                        showError('Please select a hospital');
                        return false;
                    }
                    break;
                case 2:
                    if (!$('#speciality').val()) {
                        showError('Please select a speciality');
                        return false;
                    }
                    break;
                case 3:
                    if (!$('#doctor_id').val()) {
                        showError('Please select a doctor');
                        return false;
                    }
                    break;
                case 4:
                    if (!$('#appointment_date').val()) {
                        showError('Please select a date');
                        return false;
                    }
                    break;
                case 5:
                    if (!$('#time_slot').val()) {
                        showError('Please select a time slot');
                        return false;
                    }
                    break;
                case 6:
                    if (!$('#patient_name').val() || !$('#patient_email').val() || !$('#patient_phone').val()) {
                        showError('Please fill all required fields');
                        return false;
                    }
                    break;
            }
            return true;
        }
        
        function updateStepIndicator(step, status) {
            // Update the specific step indicator based on the actual HTML structure
            // The step indicators are positioned in the DOM as siblings
            const indicators = $('.flex.flex-col.items-center.min-w-\[60px\]');
            
            if (status === 'completed') {
                // Mark this step and all previous steps as completed
                for (let i = 0; i < step; i++) {
                    const indicator = $(indicators[i]);
                    // Change background to green gradient and text to white for completed
                    indicator.find('div').first().removeClass('bg-white border-gray-300 text-gray-400').addClass('bg-gradient-to-r from-green-500 to-green-600 text-white');
                    indicator.find('span').removeClass('text-gray-500').addClass('text-green-600 font-semibold');
                    indicator.find('.indicator').removeClass('bg-white border-2 border-gray-300 text-gray-400 flex items-center justify-center mb-2 shadow transition-all').addClass('bg-gradient-to-r from-primary-500 to-primary-600 text-white flex items-center justify-center mb-2 shadow-lg');
                }
            } else if (status === 'active') {
                // Mark this step as currently active
                const indicator = $(indicators[step-1]);
                indicator.find('div').first().removeClass('bg-white border-gray-300 text-gray-400').addClass('bg-gradient-to-r from-primary-500 to-primary-600 text-white');
                indicator.find('span').removeClass('text-gray-500').addClass('text-gray-700 font-semibold');
            }
            
            // Update progress bar based on completed steps
            // Count how many have the green gradient background
            let completedSteps = 0;
            indicators.each(function(index) {
                if ($(this).find('div').first().hasClass('from-green-500') || $(this).find('div').first().hasClass('from-primary-500')) {
                    completedSteps++;
                }
            });
            
            const progress = (completedSteps / (totalSteps - 1)) * 100;
            $('#progress-bar').css('width', progress + '%');
        }
        
        function goToStep(step) {
            if (step < 1 || step > totalSteps) return;
            
            // Hide current step
            $(`#step-${currentStep}`).removeClass('active').addClass('hidden');
            
            // Show new step
            currentStep = step;
            $(`#step-${currentStep}`).removeClass('hidden').addClass('active');
            
            // Update progress bar
            const progress = (currentStep - 1) / (totalSteps - 1) * 100;
            $('#progress-bar').css('width', progress + '%');
            
            // Update step indicators
            $('.step-indicator').removeClass('active completed');
            for (let i = 1; i <= totalSteps; i++) {
                if (i < currentStep) {
                    $(`.step-indicator:eq(${i-1})`).addClass('completed');
                } else if (i === currentStep) {
                    $(`.step-indicator:eq(${i-1})`).addClass('active');
                }
            }
            
            // Update buttons
            if (currentStep === 1) {
                $('#prev-btn').addClass('hidden');
            } else {
                $('#prev-btn').removeClass('hidden');
            }
            
            if (currentStep === totalSteps) {
                $('#next-btn').addClass('hidden');
                $('#submit-btn').removeClass('hidden');
            } else {
                $('#next-btn').removeClass('hidden');
                $('#submit-btn').addClass('hidden');
            }
        }
        
        function submitAppointment() {
            if (!validateStep(6)) return;
            
            $.ajax({
                url: '<?php echo base_url(); ?>frontend/create_appointment',
                method: 'POST',
                data: $('#appointment-form').serialize(),
                dataType: 'json',
                timeout: 15000, // 15 second timeout
                beforeSend: function() {
                    showLoading();
                },
                success: function(response) {
                    hideLoading();
                    if (response && response.status === 'success') {
                        $('#success-message').html(`
                            Your appointment has been booked successfully!<br><br>
                            <strong>Doctor:</strong> ${response.doctor_name}<br>
                            <strong>Hospital:</strong> ${response.hospital_name}<br>
                            <strong>Date:</strong> ${response.appointment_date}<br>
                            <strong>Time:</strong> ${response.time_slot}
                        `);
                        $('#success-modal').removeClass('hidden');
                    } else {
                        showError(response.message || 'Failed to book appointment. Please try again.');
                    }
                },
                error: function(xhr, status, error) {
                    hideLoading();
                    console.error('AJAX Error:', xhr, status, error);
                    if (status === 'timeout') {
                        showError('Request timed out. Please check your internet connection and try again.');
                    } else if (xhr.status === 404) {
                        showError('Server endpoint not found. Please contact support.');
                    } else {
                        showError('Network error occurred. Please check your internet connection and try again.');
                    }
                }
            });
        }
        
        function showLoading() {
            $('#loading-overlay').removeClass('hidden');
        }
        
        function hideLoading() {
            $('#loading-overlay').addClass('hidden');
        }
        
        function showError(message) {
            $('#error-message').text(message);
            $('#error-modal').removeClass('hidden');
        }
    </script>
</body>
</html>
